package io.smallrye.reactive.messaging.mqtt.server;

import static io.netty.handler.codec.mqtt.MqttQoS.AT_LEAST_ONCE;
import static io.netty.handler.codec.mqtt.MqttQoS.EXACTLY_ONCE;
import static io.smallrye.reactive.messaging.mqtt.server.i18n.MqttServerLogging.log;
import static io.vertx.mqtt.MqttServerOptions.DEFAULT_PORT;
import static io.vertx.mqtt.MqttServerOptions.DEFAULT_TLS_PORT;

import java.util.concurrent.CompletableFuture;

import org.eclipse.microprofile.reactive.streams.operators.PublisherBuilder;
import org.eclipse.microprofile.reactive.streams.operators.ReactiveStreams;

import io.smallrye.mutiny.Multi;
import io.smallrye.mutiny.Uni;
import io.smallrye.mutiny.operators.multi.processors.UnicastProcessor;
import io.vertx.mqtt.MqttServerOptions;
import io.vertx.mutiny.core.Context;
import io.vertx.mutiny.core.Vertx;
import io.vertx.mutiny.mqtt.MqttServer;

class MqttServerSource {

    private final boolean broadcast;
    private final PublisherBuilder<MqttMessage> source;
    private final MqttServer mqttServer;

    private static MqttServerOptions mqttServerOptions(MqttServerConnectorIncomingConfiguration config) {
        final MqttServerOptions options = new MqttServerOptions();
        options.setAutoClientId(config.getAutoGeneratedClientId());
        options.setSsl(config.getSsl());
        // TODO set KeyCertOptions if SSL, c.f. https://vertx.io/docs/vertx-mqtt/java/#_handling_client_connection_disconnection_with_ssl_tls_support
        options.setMaxMessageSize(config.getMaxMessageSize());
        options.setTimeoutOnConnect(config.getTimeoutOnConnect());
        options.setReceiveBufferSize(config.getReceiveBufferSize());
        final int defaultPort = options.isSsl() ? DEFAULT_TLS_PORT : DEFAULT_PORT;
        options.setPort(config.getPort().orElse(defaultPort));
        options.setHost(config.getHost());
        return options;
    }

    MqttServerSource(Vertx vertx, MqttServerConnectorIncomingConfiguration config) {
        this.broadcast = config.getBroadcast();
        final MqttServerOptions options = mqttServerOptions(config);
        this.mqttServer = MqttServer.create(vertx, options);
        final UnicastProcessor<MqttMessage> processor = UnicastProcessor.create();

        mqttServer.exceptionHandler(error -> {
            log.exceptionThrown(error);
            processor.onError(error);
        });

        mqttServer.endpointHandler(endpoint -> {
            log.requestToConnect(endpoint.clientIdentifier(), endpoint.isCleanSession());

            if (endpoint.auth() != null) {
                log.requestToConnectUserName(endpoint.auth().getUsername(), endpoint.auth().getPassword());
            }
            if (endpoint.will() != null) {
                log.requestToConnectWill(endpoint.will().getWillTopic(), endpoint.will().getWillMessageBytes(),
                        endpoint.will().getWillQos(), endpoint.will().isWillRetain());
            }

            log.requestToConnectKeepAlive(endpoint.keepAliveTimeSeconds());

            endpoint.exceptionHandler(
                    error -> log.errorWithClient(endpoint.clientIdentifier(), error));

            endpoint.disconnectHandler(
                    () -> log.clientDisconnected(endpoint.clientIdentifier()));

            endpoint.pingHandler(
                    () -> log.pingReceived(endpoint.clientIdentifier()));

            endpoint.publishHandler(message -> {
                final Context ctx = vertx.getOrCreateContext();
                log.receivedMessageFromClient(message.payload(), message.qosLevel(), endpoint.clientIdentifier());

                processor.onNext(new MqttMessage(message, endpoint.clientIdentifier(), () -> {
                    CompletableFuture<Void> future = new CompletableFuture<>();
                    ctx.runOnContext(() -> {
                        if (message.qosLevel() == AT_LEAST_ONCE) {
                            log.sendToClient("PUBACK", endpoint.clientIdentifier(), message.messageId());
                            endpoint.publishAcknowledge(message.messageId());
                        } else if (message.qosLevel() == EXACTLY_ONCE) {
                            log.sendToClient("PUBREC", endpoint.clientIdentifier(), message.messageId());
                            endpoint.publishReceived(message.messageId());
                        }
                        future.complete(null);
                    });
                    return future;
                }));
            });

            endpoint.publishReleaseHandler(messageId -> {
                log.sendToClient("PUBCOMP", endpoint.clientIdentifier(), messageId);
                endpoint.publishComplete(messageId);
            });

            endpoint.subscribeHandler(subscribeMessage -> {
                log.receivedSubscription(subscribeMessage, endpoint.clientIdentifier());
                endpoint.close();
            });

            // accept connection from the remote client
            // this implementation doesn't keep track of sessions
            endpoint.accept(false);
        });

        Multi<MqttServer> server = startServer(options).cache();

        this.source = ReactiveStreams.fromPublisher(processor
                .onSubscribe().call(() -> server.collect().first().map(x -> null))
                .onSubscribe().invoke(log::newSubscriberAdded));
    }

    private Multi<MqttServer> startServer(MqttServerOptions options) {
        return mqttServer.listen()
                .onItem().invoke(s -> log.serverListeningOn(options.getHost(), s.actualPort()))
                .onFailure().invoke(log::failedToStart)
                .toMulti()
                .stage(flow -> {
                    if (broadcast) {
                        return flow.broadcast().toAllSubscribers();
                    } else {
                        return flow;
                    }
                });
    }

    synchronized PublisherBuilder<MqttMessage> source() {
        return source;
    }

    synchronized void close() {
        mqttServer.close()
                .onFailure().invoke(log::exceptionWhileClosing)
                .onItem().invoke(x -> log.closed())
                .onFailure().recoverWithUni(Uni.createFrom().nullItem())
                .await().indefinitely();
    }

    synchronized int port() {
        return mqttServer.actualPort();
    }
}
