# Connector Contribution Guide

A connector implementation is a CDI-managed bean, typically an `@ApplicationScoped` bean,
which is identified by the `@Connector` identifier.
In order to provide inbound and outbound channels, the connector implements two interfaces `InboundConnector` and `OutboundConnector` respectively.
In addition to that, the connector bean is annotated with `@ConnectorAttribute`, which describes attributes to configure channels.

!!! abstract "Maven Archetype"
    Smallrye Reactive Messaging provides a Maven archetype to bootstrap a new connector.
    You can generate a new connector project with the code described in this guide using:
    ```shell
    mvn -N archetype:generate \
    -DarchetypeGroupId=io.smallrye.reactive \
    -DarchetypeArtifactId=smallrye-reactive-messaging-connector-archetype \
    -DarchetypeVersion={{ attributes['project-version'] }} \
    -DgroupId=io.smallrye.reactive \
    -Dpackage=io.smallrye.reactive.messaging.my \
    -Dversion={{ attributes['project-version'] }} \
    -DartifactId=smallrye-reactive-messaging-my \
    -DconnectorName=my
    ```
The following is an example of a connector skeleton :

``` java
{{ insert('connectors/MyConnector.java') }}
```

Note that the `getPublisher` and `getSubscriber` methods receive MicroProfile Config `Config` instance and wrap it with
`MyConnectorIncomingConfiguration` and `MyConnectorOutgoingConfiguration` objects.

These custom channel configuration types ease getting channel configuration, including the optional or default values.
They are generated by the `smallrye-connector-attribute-processor` annotation processor, and can be configured in project pom like the following:

```xml
    <dependencies>
      <dependency>
        <groupId>${project.groupId}</groupId>
        <artifactId>smallrye-reactive-messaging-provider</artifactId>
        <version>${project.version}</version>
      </dependency>
      <dependency>
        <groupId>io.smallrye.reactive</groupId>
        <artifactId>smallrye-connector-attribute-processor</artifactId>
        <version>${project.version}</version>
      </dependency>
    </dependencies>
    <build>
      <plugins>
        <plugin>
          <groupId>org.apache.maven.plugins</groupId>
          <artifactId>maven-compiler-plugin</artifactId>
          <configuration>
            <generatedSourcesDirectory>${project.build.directory}/generated-sources/</generatedSourcesDirectory>
            <annotationProcessors>
              <annotationProcessor>
                io.smallrye.reactive.messaging.connector.ConnectorAttributeProcessor
              </annotationProcessor>
              <annotationProcessor>
                org.jboss.logging.processor.apt.LoggingToolsProcessor
              </annotationProcessor>
            </annotationProcessors>
          </configuration>
        </plugin>
      </plugins>
    </build>
```

The `smallrye-reactive-messaging-provider` is the minimum required dependency for a connector implementation.
You'll also note that the `LoggingToolsProcessor` annotation processor is also configured.
This enables generating internationalized log statements and exceptions.
Typically, you would create following interfaces in `i18n` sub-package: `[Connector]Exceptions`, `[Connector]Logging` and `[Connector]Messages`.
More information can be found in [JBoss Logging Tools documentation](https://jboss-logging.github.io/jboss-logging-tools/).

## Implementing Inbound Channels

The `InboundConnector` implementation returns, for a given channel configuration, a reactive stream of `Message`s.
The returned reactive stream is an instance of `Flow.Publisher` and typically can be implemented using Mutiny `Multi` type.

!!! note "`IncomingConnectorFactory`"
    The inbound channels can also implement the `IncomingConnectorFactory` from the MicroProfile Reactive Messaging specification.
    However, the `PublisherBuilder` type can be more challenging to work with and
    Smallrye Reactive Messaging converts the provided stream to Mutiny types to do the wiring anyway.

The returned `Flow.Publisher` would allow controlling the flow of ingestion using backpressure.
It would be preferable to use pull-based APIs of the underlying messaging library to receive messages from the message broker.
You can refer to the [`Mutiny` "How to use polling?" guide](https://smallrye.io/smallrye-mutiny/latest/guides/polling/)
to construct a `Multi` using Mutiny APIs, or implement the `Flow.Subscription` from scratch and wrap it in an `AbstractMulti`.

Here is an example channel implementation which constructs the reactive stream using the polling API:
``` java
{{ insert('connectors/MyIncomingChannel.java') }}
```

### Connector Threading and Vert.x

Whether the external API call is blocking or non-blocking, managing the thread on which the message processing will be dispatched can be challenging.
Smallrye Reactive Messaging depends on [Eclipse Vert.x](https://vertx.io/) to consistently dispatch messages in event-loop or worker threads,
propagating the message context along different processing stages.
You can read more on [Message Context](message-context.md) and [Vert.x Context](https://vertx.io/blog/an-introduction-to-the-vert-x-context-object/).

Some connectors already use Vert.x clients, such as RabbitMQ, AMQP 1.0 or MQTT.
Other connectors such as Kafka or Pulsar directly use the client library of the messaging technology,
therefore they create a Vert.x Context per channel to dispatch messages on that context.
Connectors can access the `Vertx` instance by injecting the `ExecutionHolder` bean.
[Mutiny operators `runSubscribtionOn` and `emitOn`](https://smallrye.io/smallrye-mutiny/latest/guides/emit-on-vs-run-subscription-on/) can be used to switch threads the events are dispatched on.

### Custom Message and Metadata implementations

Reactive messaging {{ javadoc('org.eclipse.microprofile.reactive.messaging.Message', False, 'io.smallrye.reactive/smallrye-reactive-messaging-api') }}
type is a thin wrapper around a payload and some metadata, which lets implementing acknowledgment and negative acknowledgment of that message.

Messages received from the underlying library very often return with a custom type,
wrapping the payload and some properties such as key, timestamp, schema information, or any other metadata.

!!! tip
    The client may propose different strategies for consuming messages individually or in batches.
    If a batch consuming is available the incoming channel may receive wrap and dispatch message as a batch or individually.

While it is possible to use `Message.of` builder methods to wrap the incoming message type,
a custom type implementing `Message` interface helps to deal with different aspects we'll cover later,
such as deserialization, message acknowledgment or tracing.

An example message implementation would be like the following:
``` java
{{ insert('connectors/MyMessage.java') }}
```

Note that `MyMessage` implements the `ContextAwareMessage`.
In the constructor `captureContextMetadata` helper method is used to capture the Vert.x context which created the object and capturing it into the `LocalContextMetadata`.
This metadata allows running [each message in its own Vert.x context, supporting context propagation](message-context.md).

The `MyMessage` type implements the accessors for the `metadata` and `payload` from the `Message` interface.
If the messaging technology doesn't have a built-in unmarshalling mechanism, the message can deserialize the raw payload to a primitive or a complex object.

!!! warning
    The custom message implementation is usually not the type consumed by the application injecting channels.
    Applications usually inject in the payload, the raw consumed type (in the above example the `ConsumedMessage`),
    or some other type provided by the [`MessageConverter`s](converters.md).
    Handling of `Message` types by the application is restricted only to advanced use cases, because handling of message acknowledgment is manual
    Even then the message may be intercepted before and changed, conserving the `metadata`, `ack` and `nack` handlers but not the original type created by the connector.

The `MyIncomingMetadata` gives access to the underlying consumed message attributes, and applications can inject this object for accessing message details:
``` java
{{ insert('connectors/MyIncomingMetadata.java') }}
```

Also note that `ack` and `nack` method implementations are delegated to handler objects.
This allows configuring different strategies at channel level.

### Acknowledgment strategies

The [acknowledgement](acknowledgement.md) is the way for message consumers to inform the broker that the message has been successfully received and processed.
Depending on the messaging technology the broker then can decide to remove the message from the queue, flag as consumed or purge it completely.
In Reactive Messaging there are different policies to trigger the acknowledgement but the canonical one is to acknowledge a message when the processing (potentially asynchronous) has completed (`POST_PROCESSING`).

The Reactive Messaging defines `Message#ack` method as non-blocking asynchronous, returning a `CompletionStage<Void>`,
because potentially the acknowledgement action sends a network call to the broker.

The following example simply calls the client `ack` method using the Mutiny `Uni` and switch the emitter to the Message context.
Returning back to the message context is essential for chaining asynchronous actions without losing the context and for keeping the consistency on message consumption flow.

``` java
{{ insert('connectors/MyAckHandler.java') }}
```

While this ack handler strategy acknowledges each message to the broker,
the messaging technology can allow employing different strategies for acknowledging messages.
For example an ack strategy can track processed messages and acknowledge them altogether or call a different client side endpoint to acknowledge the message batch.

### Failure handling strategies

The failure handling, or the [negative acknowledgment](acknowledgement.md#negative-acknowledgement) allows indicating that a message was not processed correctly.
Similar to the acknowledgment the Reactive Messaging defines `Message#nack(Throwable reason, Metadata metadata)` method as non-blocking asynchronous, returning a `CompletionStage<Void>`.

``` java
{{ insert('connectors/MyFailureHandler.java') }}
```

Different failure handling strategies can, for example,
- Ignore the failure, log and call the `ack` instead
- Send the message to a dead letter queue and call the `ack`
- Employ a different strategy depending on the `Metadata` associated with the `nack` method call.

### Message Converters

The connector can propose default [`MessageConverter`](converters.md) implementations for converting the payload to a custom type.
As an example the following converter extracts the `CustomMessage` and puts it in the payload:
``` java
{{ insert('connectors/MyMessageConverter.java') }}
```

## Implementing Outbound Channels

The `OutboundConnector` implementation returns, for a given channel configuration, a `Flow.Subscriber` of messages.
This is typically implemented by a custom `Flow.Processor` and using the `MultiUtils.via` helper methods to apply message transformations.

!!! note "`OutgoingConnectorFactory`"
    The outbound channels can also implement the `OutgoingConnectorFactory` from the MicroProfile Reactive Messaging specification.
    However, it is usually more friendly to work with the `MultiUtils.via` methods to construct and transform outgoing messages.

Here is an example outgoing channel implementation:
``` java
{{ insert('connectors/MyOutgoingChannel.java') }}
```
The `MultiUtils.via` helper method allows using the `Multi` chaining methods and in the same time provides a `Flow.Subscriber`.
However, this implementation allows sending messages one at a time:
one only after the previous outgoing message send is completed.

Some messaging technologies provide publish receipt,
a message back from the broker to the sender that asynchronously acknowledges the sent operation.
In this case the connector can only be sure of the send operation when it receives the publish receipt of that message.
Some technologies may provide blocking sending calls,
in that case the connector needs to delegate the sending call to a worker thread.

Depending on whether the client supports multiple in-flight outgoing messages, you can also use a `SenderProcessor`,
which allows receiving configuration for the maximum number of in-flight messages and whether it waits for completion (publish receipt from the broker):

``` java
{{ insert('connectors/MyOutgoingChannelWithPartials.java', 'sender-processor') }}
```

Other more advanced scenarios can be implemented to retry the transmission in case of a retryable failure,
or batch multiple outgoing messages to a single send operation.

### Outgoing `Message` Builder

In order to convey all attributes of the outgoing message to the client library
connectors provide outgoing `Message` implementation and a corresponding outgoing message metadata.
These allow the application developer to build the message attributes that will be sent to the broker.

``` java
{{ insert('connectors/MyOutgoingMessage.java') }}
```

``` java
{{ insert('connectors/MyOutgoingMetadata.java') }}
```

The outgoing channel implementation then will construct the client library object that represents the outbound message, `SendMessage` in this example:

``` java
{{ insert('connectors/MyOutgoingChannelWithPartials.java', 'send-message') }}
```

It is a best practice to also allow the application to return a payload of the client outbound library object (`SendMessage`).

### Outgoing message acknowledgement

Because the Reactive Messaging [chains acknowledgements](acknowledgement.md#chain-of-acknowledgment) from incoming message until the outgoing message,
it is crucial for the outgoing channel to correctly ack and nack the message.

## Smallrye Health Integration

Smallrye Reactive Messaging allows connectors to integrate with Smallrye Health to contribute channel state to the health reports.
Connectors need to implement the `HealthReporter` interface and implement some or all of the `getReadiness`, `getLiveness` and `getStartup` methods:

``` java
{{ insert('connectors/MyConnectorWithPartials.java', 'health-report') }}
```

Implementing health reports per channel depends on what information is available to the connector.
For more information on different health check probes you can check out [Configure Liveness, Readiness and Startup Probes
](https://kubernetes.io/docs/tasks/configure-pod-container/configure-liveness-readiness-startup-probes/)

You may want to add a connector attribute to enable/disable the health reporting per channel:
```java
import io.smallrye.reactive.messaging.annotations.ConnectorAttribute;

@ConnectorAttribute(name = "health-enabled", type = "boolean", direction = ConnectorAttribute.Direction.INCOMING_AND_OUTGOING, description = "Whether health reporting is enabled (default) or disabled", defaultValue = "true")
```

## OpenTelemetry Tracing Integration

Smallrye Reactive Messaging allows connectors to easily integrate with OpenTelemetry tracing.
It propagates the tracing context from inbound messages and to outbound messages.
The `smallrye-reactive-messaging-otel` module provides necessary dependencies to the OpenTelemetry artifacts and also provides `TracingUtils` helper class for setting up the tracing.

```xml
<dependency>
    <groupId>io.smallrye.reactive.messaging</groupId>
    <artifactId>smallrye-reactive-messaging-otel</artifactId>
</dependency>
```

For integrating tracing you'd need to create a couple of classes:

- Holder class for tracing information
- Implementation of `io.opentelemetry.context.propagation.TextMapGetter`, which retrieves for a given key the value of a tracing attributed from the holder object
- Implementation of `io.opentelemetry.context.propagation.TextMapSetter`, which sets on the holder object the key and value of tracing attributes
- Implementations of `io.opentelemetry.instrumentation.api.instrumenter.AttributesExtractor` and `io.opentelemetry.instrumentation.api.instrumenter.messaging.MessagingAttributesGetter`

Then you'd need to configure instrumenters per incoming and outgoing channel:

``` java
{{ insert('connectors/tracing/MyOpenTelemetryInstrumenter.java', 'create-instrumenter') }}
```

Finally, you'd need to configure instrumenters per incoming and outgoing channels and wire the call to instrumenter using `TracingUtils`.

For an incoming channel, you'd need to call the instrumenter on an inbound message:

``` java
{{ insert('connectors/MyIncomingChannelWithPartials.java', 'incoming-tracing') }}
```

For an outgoing channel, you'd need to call the instrumenter on constructing the outbound message:

``` java
{{ insert('connectors/MyOutgoingChannelWithPartials.java', 'outgoing-tracing') }}
```

You may want to add a connector attribute to enable/disable the tracing per channel:
```java
import io.smallrye.reactive.messaging.annotations.ConnectorAttribute;

@ConnectorAttribute(name = "tracing-enabled", type = "boolean", direction = ConnectorAttribute.Direction.INCOMING_AND_OUTGOING, description = "Whether tracing is enabled (default) or disabled", defaultValue = "true")
```

## Testing the connector

While unit tests are highly encouraged for validating ad-hoc logic in connector code,
by nature connector tests are mostly integration tests validating the correct configuration and functioning of channels.
Most of the time tests need to run against a broker instance.
This instance can be mocked or embedded in the test JVM,
or provisioned in a container runtime using [Testcontainers](https://testcontainers.org).
The testcontainers approach is encouraged as it'll provide a testing environment closest to reality.

It may take too much time and resources to start a broker per test method or per test class,
so may want to share the same broker instance between all test classes.
In that case you can checkout how to write a [JUnit 5 Extension](https://junit.org/junit5/docs/current/user-guide/#extensions)
and start only one container instance in the beginning of tests and stop it at the end of all the tests.

There are essentially two ways of creating the connector behavior to test against:

1. Instantiating channels directly by passing the custom configuration.
With this you can get the Reactive stream directly from the channel implementation and send/receive messages.
You can use [AssertSubscriber from Mutiny](https://smallrye.io/smallrye-mutiny/2.5.1/guides/testing/) to regulate demand and write assertions.
2. CDI-based tests which write configuration and instantiate application beans.
You can use [Weld, the reference implementation of CDI specification](https://weld.cdi-spec.org/) with configured set of beans and extensions:
``` java
{{ insert('connectors/test/WeldTestBase.java') }}
```

You would need following test dependencies for enabling Weld in tests:
```xml
    <dependency>
      <groupId>io.smallrye.reactive</groupId>
      <artifactId>test-common</artifactId>
      <scope>test</scope>
    </dependency>
    <dependency>
      <groupId>org.jboss.weld.se</groupId>
      <artifactId>weld-se-shaded</artifactId>
      <version>${weld.version}</version>
      <scope>test</scope>
    </dependency>
    <dependency>
      <groupId>org.jboss.weld</groupId>
      <artifactId>weld-core-impl</artifactId>
      <version>${weld.version}</version>
      <scope>test</scope>
    </dependency>
    <dependency>
      <groupId>org.awaitility</groupId>
      <artifactId>awaitility</artifactId>
      <scope>test</scope>
    </dependency>
```

Your test classes can therefore extend the `WeldTestBase` and provide configuration and application beans:

``` java
{{ insert('connectors/test/MyConnectorTest.java') }}
```

!!! tip "Awaitility"
    Because connector tests are usually asynchronous, [awaitility](https://github.com/awaitility/awaitility)
    provides a DSL to await on expressed assertions.

### Common tests for validating the connector

- Message consumption through incoming channels
- Message producing through outgoing channels
- Ack and failure handler strategies test
- Message Context propagation test `LocalPropagationTest`
- `HealthCheckTest`
- `MessageConverterTest`
- `TracingPropagationTest`
- Configuration test
- Authentication test
- Tests for
