package io.smallrye.reactive.messaging.kafka.commit;

import static org.assertj.core.api.Assertions.assertThat;
import static org.awaitility.Awaitility.await;

import java.time.Duration;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import jakarta.enterprise.context.ApplicationScoped;

import org.apache.kafka.clients.consumer.*;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.eclipse.microprofile.reactive.messaging.Incoming;
import org.junit.jupiter.api.Test;

import io.smallrye.mutiny.Uni;
import io.smallrye.reactive.messaging.kafka.base.KafkaCompanionTestBase;
import io.smallrye.reactive.messaging.test.common.config.MapBasedConfig;

public class PartitionTest extends KafkaCompanionTestBase {

    @Test
    public void testWithPartitions() {
        companion.topics().createAndWait(topic, 3);
        String groupId = UUID.randomUUID().toString();

        MapBasedConfig config = kafkaConfig("mp.messaging.incoming.kafka")
                .with("group.id", groupId)
                .with("topic", topic)
                .with("concurrency", 3)
                .with("auto.offset.reset", "earliest")
                .with("value.deserializer", StringDeserializer.class.getName());

        MyApplication application = runApplication(config, MyApplication.class);

        int expected = 3000;
        Random random = new Random();
        companion.produceStrings().usingGenerator(i -> {
            int p = random.nextInt(3);
            return new ProducerRecord<>(topic, p, Integer.toString(p), Integer.toString(i));
        }, expected).awaitCompletion(Duration.ofMinutes(1));

        await()
                .atMost(60, TimeUnit.SECONDS)
                .until(() -> application.count() >= expected);
        assertThat(application.getReceived().keySet()).hasSizeGreaterThanOrEqualTo(getMaxNumberOfEventLoop(3));

        await().until(() -> {
            Map<TopicPartition, OffsetAndMetadata> map = companion.consumerGroups().offsets(groupId);
            long c = map.values().stream().map(OffsetAndMetadata::offset).mapToLong(l -> l).sum();
            return map.size() == 3 && c == expected;
        });
    }

    @Test
    public void testWithMoreConsumersThanPartitions() {
        companion.topics().createAndWait(topic, 3);
        String groupId = UUID.randomUUID().toString();
        MapBasedConfig config = kafkaConfig("mp.messaging.incoming.kafka")
                .with("group.id", groupId)
                .with("topic", topic)
                .with("concurrency", 5) // 2 idles
                .with("auto.offset.reset", "earliest")
                .with("value.deserializer", StringDeserializer.class.getName());

        MyApplication application = runApplication(config, MyApplication.class);

        int expected = 3000;
        Random random = new Random();
        companion.produceStrings().usingGenerator(i -> {
            int p = random.nextInt(3);
            return new ProducerRecord<>(topic, p, Integer.toString(p), Integer.toString(i));
        }, expected).awaitCompletion(Duration.ofMinutes(1));

        await()
                .atMost(60, TimeUnit.SECONDS)
                .until(() -> application.count() >= expected);
        assertThat(application.getReceived().keySet()).hasSizeGreaterThanOrEqualTo(getMaxNumberOfEventLoop(3));

        await().until(() -> {
            Map<TopicPartition, OffsetAndMetadata> map = companion.consumerGroups().offsets(groupId);
            long c = map.values().stream().map(OffsetAndMetadata::offset).mapToLong(l -> l).sum();
            return map.size() == 3 && c == expected;
        });
    }

    @Test
    public void testWithMorePartitionsThanConsumers() {
        companion.topics().createAndWait(topic, 3);
        String groupId = UUID.randomUUID().toString();

        MapBasedConfig config = kafkaConfig("mp.messaging.incoming.kafka")
                .with("group.id", groupId)
                .with("topic", topic)
                .with("concurrency", 2) // one consumer will get 2 partitions
                .with("auto.offset.reset", "earliest")
                .with("value.deserializer", StringDeserializer.class.getName());

        MyApplication application = runApplication(config, MyApplication.class);

        int expected = 3000;
        Random random = new Random();
        companion.produceStrings().usingGenerator(i -> {
            int p = random.nextInt(3);
            return new ProducerRecord<>(topic, p, Integer.toString(p), Integer.toString(i));
        }, expected).awaitCompletion(Duration.ofMinutes(1));

        await()
                .atMost(60, TimeUnit.SECONDS)
                .until(() -> application.count() >= expected);
        assertThat(application.getReceived().keySet()).hasSizeGreaterThanOrEqualTo(getMaxNumberOfEventLoop(2));

        await().until(() -> {
            Map<TopicPartition, OffsetAndMetadata> map = companion.consumerGroups().offsets(groupId);
            long c = map.values().stream().map(OffsetAndMetadata::offset).mapToLong(l -> l).sum();
            return map.size() == 3 && c == expected;
        });
    }

    @ApplicationScoped
    public static class MyApplication {
        private final AtomicLong count = new AtomicLong();
        private final Map<String, List<String>> received = new ConcurrentHashMap<>();

        AtomicInteger p = new AtomicInteger();

        @Incoming("kafka")
        public Uni<Void> consume(String payload) {
            String k = Thread.currentThread().getName();
            List<String> list = received.computeIfAbsent(k, s -> new CopyOnWriteArrayList<>());
            list.add(payload);
            count.incrementAndGet();
            return Uni.createFrom().voidItem().onItem().delayIt().by(Duration.ofMillis(10))
                    .invoke(() -> p.decrementAndGet());
        }

        public Map<String, List<String>> getReceived() {
            return received;
        }

        public long count() {
            return count.get();
        }
    }

}
