/*
 * Copyright (c) 2019 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at:
 *
 *     https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
package org.eclipse.jkube.kit.build.service.docker;

import org.apache.commons.io.FileUtils;
import org.eclipse.jkube.kit.build.service.docker.access.DockerAccess;
import org.eclipse.jkube.kit.build.service.docker.access.DockerAccessException;
import org.eclipse.jkube.kit.common.JKubeConfiguration;
import org.eclipse.jkube.kit.common.JavaProject;
import org.eclipse.jkube.kit.common.KitLogger;
import org.eclipse.jkube.kit.config.image.ImageConfiguration;
import org.eclipse.jkube.kit.config.image.build.BuildConfiguration;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.util.Collections;
import java.util.Properties;

import static org.assertj.core.api.Assertions.assertThatIllegalStateException;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

class BuildServiceTest {

  @TempDir
  private File tempDir;
  private DockerAccess mockedDockerAccess;
  private BuildService buildService;
  private ImageConfiguration imageConfiguration;
  private ImagePullManager mockedImagePullManager;
  private Properties projectProperties;
  private JKubeConfiguration jKubeConfiguration;
  private RegistryService mockedRegistryService;

  @BeforeEach
  void setUp() {
    mockedDockerAccess = mock(DockerAccess.class, RETURNS_DEEP_STUBS);
    ArchiveService mockedArchiveService = mock(ArchiveService.class, RETURNS_DEEP_STUBS);
    mockedRegistryService = mock(RegistryService.class, RETURNS_DEEP_STUBS);
    mockedImagePullManager = mock(ImagePullManager.class, RETURNS_DEEP_STUBS);
    projectProperties = new Properties();
    jKubeConfiguration = JKubeConfiguration.builder()
      .project(JavaProject.builder()
        .properties(projectProperties)
        .baseDirectory(tempDir)
        .build())
      .sourceDirectory(tempDir.getAbsolutePath())
      .build();
    QueryService mockedQueryService = new QueryService(mockedDockerAccess);
    buildService = new BuildService(
      mockedDockerAccess,
      mockedQueryService,
      mockedRegistryService,
      mockedArchiveService,
      new KitLogger.SilentLogger()
    );
    imageConfiguration = ImageConfiguration.builder()
        .name("image-name")
        .build(BuildConfiguration.builder()
            .from("from")
            .tags(Collections.singletonList("latest"))
            .build()
        ).build();
  }

  @Test
  void buildImage_whenValidImageConfigurationProvidedAndDockerDaemonReturnsValidId_shouldBuildImage() throws IOException {
    // Given
    when(mockedDockerAccess.getImageId("image-name")).thenReturn("c8003cb6f5db");

    // When
    buildService.buildImage(imageConfiguration, mockedImagePullManager, jKubeConfiguration);

    // Then
    verify(mockedDockerAccess, times(1))
        .buildImage(eq("image-name"), any(), any());
  }

  @Test
  void buildImage_whenValidImageConfigurationProvidedAndDockerDaemonReturnsNull_shouldBuildImage() throws IOException {
    // Given
    when(mockedDockerAccess.getImageId("image-name")).thenReturn(null);
    // When & Then
    assertThatIllegalStateException()
            .isThrownBy(() -> buildService.buildImage(imageConfiguration, mockedImagePullManager, jKubeConfiguration))
            .withMessage("Failure in building image, unable to find image built with name image-name");
  }

  @Test
  void tagImage_whenValidImageConfigurationProvided_shouldTagImage() throws DockerAccessException {
    // When
    buildService.tagImage("image-name", imageConfiguration);

    // Then
    verify(mockedDockerAccess, times(1))
        .tag("image-name", "image-name:latest", true);
  }

  @Test
  void buildImage_whenMultiStageDockerfileWithBuildArgs_shouldPrepullImages() throws IOException {
    // Given
    when(mockedDockerAccess.getImageId("image-name")).thenReturn("c8003cb6f5db");
    File multistageDockerfile = copyToTempDir("Dockerfile_multi_stage_with_args_no_default");
    imageConfiguration = ImageConfiguration.builder()
        .name("image-name")
        .build(BuildConfiguration.builder()
            .dockerFile(multistageDockerfile.getPath())
            .dockerFileFile(multistageDockerfile)
            .build())
        .build();

    projectProperties.setProperty("docker.buildArg.VERSION", "latest");
    projectProperties.setProperty("docker.buildArg.FULL_IMAGE", "busybox:latest");
    projectProperties.setProperty("docker.buildArg.REPO_1", "docker.io/library");
    projectProperties.setProperty("docker.buildArg.IMAGE-1", "openjdk");

    // When
    buildService.buildImage(imageConfiguration, mockedImagePullManager, jKubeConfiguration);

    // Then
    verify(mockedRegistryService, times(1)).pullImageWithPolicy(eq("fabric8/s2i-java:latest"), any(), any(), any());
    verify(mockedRegistryService, times(1)).pullImageWithPolicy(eq("busybox:latest"), any(), any(), any());
    verify(mockedRegistryService, times(1)).pullImageWithPolicy(eq("docker.io/library/openjdk:latest"), any(), any(),
        any());
  }

  private File copyToTempDir(String resource) throws IOException {
    File ret = new File(tempDir, "Dockerfile");
    try (OutputStream os = Files.newOutputStream(ret.toPath())) {
      FileUtils.copyFile(new File(getClass().getResource(resource).getPath()), os);
    }
    return ret;
  }
}
