package com.redhat.installer.action;

import com.izforge.izpack.event.SimpleInstallerListener;
import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.util.AbstractUIProgressHandler;
import com.izforge.izpack.util.CleanupClient;
import com.izforge.izpack.util.Housekeeper;
import com.redhat.installer.util.ValidatorConstants;
import org.apache.commons.io.FileUtils;

import java.io.*;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.Queue;

public class BackupListener extends SimpleInstallerListener implements CleanupClient {

	private static AutomatedInstallData idata;
    private static String installPath;
    private static String backupPath;

	/**
	 * Restores backups of all files as needed. TODO: perhaps refactor to use a
	 * list of somekind, defined in a Constants class
	 */
	public void cleanUp() {
        String aborted = idata.getVariable("install.aborted");
        File backupFolder = new File(backupPath);
        File installFolder = new File(installPath);

        if (backupFolder.exists()){
            if (aborted != null && Boolean.parseBoolean(aborted)) {
                try {
                    FileUtils.deleteDirectory(installFolder);
                } catch(IOException e) {
                    e.printStackTrace();
                }
                /**
                 * Restore the backed up eap installation.
                 */
                recursiveDirectoryCopy(backupPath, installPath);
            }

            /**
             * Now remove the backup folder as it's not needed anymore.
             */
            try {
                FileUtils.deleteDirectory(backupFolder);
            } catch(IOException e) {
                e.printStackTrace();
            }
        }
	}

    /**
     * Pre-order traversal of the file system rooted at rootPath while re-creating
     * the file structure but rooted at rootDestPath.
     * @param rootPath
     * @param rootDestPath
     */
    private void recursiveDirectoryCopy(String rootPath, String rootDestPath) {
        Queue<File> files = new LinkedList<File>();
        File root = new File(rootPath);
        files.add(root);

        while (!files.isEmpty()) {
            File current = files.remove();
            File backup =  new File(current.getPath().replace(rootPath,rootDestPath));

            if (current.isDirectory()) {
                for (File child : current.listFiles()) {
                    files.add(child);
                }
                backup.mkdir();
            }

            copyRetainingPermissions(current, backup);
        }
    }

    /**
     * Attempts to retain read/write/exec permissions when copying files from source to destination.
     * @param source
     * @param dest
     */
    private void copyRetainingPermissions(File source, File dest) {
        boolean exec = source.canExecute();
        boolean read = source.canRead();
        boolean write = source.canWrite();

        try {
            if (source.isDirectory()) {
                FileUtils.forceMkdir(dest);
            } else {
                FileUtils.copyFile(source, dest, true);
            }
            dest.setExecutable(exec);
            dest.setReadable(read);
            dest.setWritable(write);
        } catch(IOException e){
            // Throws if it's copying a dir
        }

    }

	@Override
	public void beforePacks(AutomatedInstallData idata, Integer npacks, AbstractUIProgressHandler handler) throws Exception {
		// just to save time, since izpack doesn't have a good
		// "execute this before any installation occurs, but only once."
		if (this.idata == null) {
			this.idata = idata;
            installPath = idata.getVariable("INSTALL_PATH") + File.separator + idata.getVariable("INSTALL_SUBPATH");
            backupPath = idata.getVariable("INSTALL_PATH") + File.separator + idata.getVariable("INSTALL_SUBPATH") + ValidatorConstants.backupExt;
		}
        Housekeeper.getInstance().registerForCleanup(this);
        File installPathFile = new File(installPath);

        /**
         * If an existing installation is detected:
         */
        if (installPathFile.exists()) {
            /**
             * Back up the entire jboss-eap-6.* directory, so we can restore it without
             * manging the former installation in the event of some dastardly user quitting early.
             */
            recursiveDirectoryCopy(installPath, backupPath);
        }
    }
	
	
	@Override
	public void afterPacks(AutomatedInstallData idata, AbstractUIProgressHandler handler) throws Exception {


        // Nothing as of yet


	}
}
