package com.redhat.installer.layering.validator;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.DataValidator;
import com.izforge.izpack.util.Debug;
import org.apache.commons.lang.SystemUtils;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

/**
 * Class that evaluates the directory the user provides is a valid platform.
 * Moderate rewrite and update of EapExistsValidator
 * <p>
 * Current valid platforms:
 * EAP 7.x.y
 * Tomcat8 3.1.1 - Execution server only?
 * <p>
 * To be added:
 * Weblogic
 * Websphere
 */
public class IsSupportedContainerValidator implements DataValidator {
    private static final String VERSION_REGEX = "\\b\\d\\.\\d\\.\\d[\\.\\w\\w\\d]*\\b";
    private AutomatedInstallData idata;
    private String error;
    private String message;

    // EAP constants
    private static List<Path> requiredEapFiles;

    static {
        Path standaloneConfiguration = Paths.get("standalone", "configuration");
        Path domainConfiguration = Paths.get("domain", "configuration");
        Path binFolder = Paths.get("bin");
        requiredEapFiles = new ArrayList<>();
        requiredEapFiles.add(standaloneConfiguration.resolve("standalone.xml"));
        requiredEapFiles.add(standaloneConfiguration.resolve("standalone-ha.xml"));
        requiredEapFiles.add(standaloneConfiguration.resolve("standalone-full.xml"));
        requiredEapFiles.add(standaloneConfiguration.resolve("standalone-full-ha.xml"));
        requiredEapFiles.add(domainConfiguration.resolve("domain.xml"));
        requiredEapFiles.add(domainConfiguration.resolve("host.xml"));
        requiredEapFiles.add(binFolder.resolve("add-user.sh"));
        requiredEapFiles.add(Paths.get("version.txt"));
    }

    private final Path eapDeployments = Paths.get("standalone", "deployments");
    private List<Path> eapProductFiles;

    // Tomcat8 constants
    private static List<Path> requiredTomcatFiles;

    static {
        // should ensure that version.sh OR version.bat exists
        Path configurationFolder = Paths.get("conf");
        requiredTomcatFiles = new ArrayList<>();
        requiredTomcatFiles.add(Paths.get("lib/"));
        requiredTomcatFiles.add(Paths.get("bin/"));
        requiredTomcatFiles.add(configurationFolder.resolve("context.xml"));
        requiredTomcatFiles.add(configurationFolder.resolve("server.xml"));
        requiredTomcatFiles.add(configurationFolder.resolve("tomcat-users.xml"));
    }

    private static final Path webappsFolder = Paths.get("webapps");
    private static List<Path> tomcatProductFiles;

    // General enums and constants
    enum Container {
        EAP, Tomcat8, Weblogic, Websphere, Unsupported
    }

    enum InstallState {
        OK, LowVersion, HighVersion, MissingVersion, ProductAlreadyInstalled, NotInstalled
    }

    enum VersionState {
        OK,
        LowVersion, // perhaps we can just use a boolean instead of high / low. helps with message to user though
        HighVersion,
        MissingVersion // is this needed?
    }


    /**
     * Returns the container discovered in the user's provided installation path.
     *
     * @return value from Container enum that indicates what the target container is.
     */
    private Container getContainerInInstallationPath() {
        Path installPath = Paths.get(idata.getVariable("INSTALL_PATH") + File.separator);
        ContainerInformation eapInfo = isEapInstalled(installPath);
        ContainerInformation tomcatInfo = isTomcatInstalled(installPath);
        ContainerInformation websphereInfo = isWebsphereInstalled(installPath);
        ContainerInformation weblogicInfo = isWeblogicInstalled(installPath);
        boolean eapInstalled = true;
        boolean tomcatInstalled = true;
        boolean websphereInstalled = true;
        boolean weblogicInstalled = true;

        switch (eapInfo.getState()) {
            case OK:
                return Container.EAP;
            case LowVersion:
                setMessage("container.low.version", eapInfo.getName(), getSupportedEapVersion(), eapInfo.getVersion().getVersionString());
                break;
            case HighVersion:
                setMessage("container.high.version", eapInfo.getName(), getSupportedEapVersion(), eapInfo.getVersion().getVersionString());
                break;
            case MissingVersion:
                setMessage("container.missing.version", eapInfo.getName());
                break;
            case ProductAlreadyInstalled:
                setMessage("container.product.already.installed");
                break;
            case NotInstalled:
                eapInstalled = false;
                break;
        }

        switch (tomcatInfo.getState()) {
            case OK:
                return Container.Tomcat8;
            case LowVersion:
                setMessage("container.low.version", tomcatInfo.getName(), getMaximumSupportedTomcatVersion(), tomcatInfo.getVersion().getVersionString());
                break;
            case HighVersion:
                setMessage("container.high.version", tomcatInfo.getName(), getMaximumSupportedTomcatVersion(), tomcatInfo.getVersion().getVersionString());
                break;
            case ProductAlreadyInstalled:
                setMessage("container.product.already.installed");
                break;
            case MissingVersion:
                setMessage("container.missing.version", tomcatInfo.getName());
                break;
            case NotInstalled:
                tomcatInstalled = false;
                break;
        }

        switch (websphereInfo.getState()) {
            case OK:
                return Container.Websphere;
            case LowVersion:
                setMessage("container.low.version", websphereInfo.getName(), "", websphereInfo.getVersion().getVersionString());
                break;
            case HighVersion:
                setMessage("container.high.version", websphereInfo.getName(), "", websphereInfo.getVersion().getVersionString());
                break;
            case ProductAlreadyInstalled:
                setMessage("container.product.already.installed");
                break;
            case MissingVersion:
                setMessage("container.missing.version", websphereInfo.getName());
                break;
            case NotInstalled:
                websphereInstalled = false;
                break;
        }

        switch (weblogicInfo.getState()) {
            case OK:
                return Container.Weblogic;
            case LowVersion:
                setMessage("container.low.version");
                break;
            case HighVersion:
                setMessage("container.high.version");
                break;
            case MissingVersion:
                setMessage("container.missing.version");
                break;
            case ProductAlreadyInstalled:
                setMessage("container.product.already.installed");
                break;
            case NotInstalled:
                weblogicInstalled = false;
                break;
        }

        if (!eapInstalled && !tomcatInstalled && !websphereInstalled && !weblogicInstalled) {
            setMessage("container.no.supported");
        }

        return Container.Unsupported;
    }

    private ContainerInformation isWeblogicInstalled(Path installPath) {
        ContainerInformation weblogicInfo = new ContainerInformation(Container.Weblogic.toString());
        weblogicInfo.setState(InstallState.NotInstalled);
        return weblogicInfo;
    }

    private ContainerInformation isWebsphereInstalled(Path installPath) {
        ContainerInformation websphereInfo = new ContainerInformation(Container.Websphere.toString());
        websphereInfo.setState(InstallState.NotInstalled);
        return websphereInfo;
    }

    // Tomcat8 methods
    private ContainerInformation isTomcatInstalled(Path tomcatHome) {
        ContainerInformation tomcatInfo = new ContainerInformation(Container.Tomcat8.toString());
        if (!containsAllPaths(tomcatHome, requiredTomcatFiles)) {
            tomcatInfo.setState(InstallState.NotInstalled);
            return tomcatInfo;
        }

        if (containsAnyPath(tomcatHome, tomcatProductFiles)) {
            tomcatInfo.setState(InstallState.ProductAlreadyInstalled);
            return tomcatInfo;
        }
        tomcatInfo.setVersion(new Version(getInstalledTomcatVersionString(tomcatHome)));
        VersionState versionCheck = isTomcatValidVersion(tomcatInfo.getVersion());
        switch (versionCheck) {
            case OK:
                tomcatInfo.setState(InstallState.OK);
                break;
            case LowVersion:
                tomcatInfo.setState(InstallState.LowVersion);
                break;
            case HighVersion:
                tomcatInfo.setState(InstallState.HighVersion);
                break;
            default:
                tomcatInfo.setState(InstallState.MissingVersion);
                break;
        }
        return tomcatInfo;
    }


    /**
     * Tests if the provided tomcatVersion is acceptable for installation
     *
     * @param tomcatVersion Version object containing the tomcat Version string to test
     * @return MissingVersion if the provided version is empty, LowVersion if the version is below the supported version, OK otherwise.
     */
    private VersionState isTomcatValidVersion(Version tomcatVersion) {
        Version supportedTomcatVersion = new Version(getMaximumSupportedTomcatVersion());

        if (tomcatVersion.getVersionString().isEmpty()) {
            return VersionState.MissingVersion;
        }
        if (tomcatVersion.compareTo(supportedTomcatVersion) < 0) {
            return VersionState.LowVersion;
        } else if (tomcatVersion.compareTo(supportedTomcatVersion) > 0) {
            return VersionState.HighVersion;
        } else {
            return VersionState.OK;
        }
    }

    private String getInstalledTomcatVersionString(Path tomcatHome) {
        Path versionExecutablePath = findTomcatVersionExecutable(tomcatHome);
        if (!Files.exists(versionExecutablePath)) {
            return "";
        }
        return extractTomcatVersion(getExecutableOutputAsLines(versionExecutablePath));
    }

    /**
     * Finds the appropriate tomcat version executable
     *
     * @param tomcatHome input TOMCAT_HOME path
     * @return Path to the appropriate version executable
     */
    private Path findTomcatVersionExecutable(Path tomcatHome) {
        String filename;
        if (SystemUtils.IS_OS_WINDOWS) {
            filename = "bin/version.bat";
        } else {
            filename = "bin/version.sh";
        }
        return tomcatHome.resolve(filename);
    }

    /**
     * Utility method that returns a Stream of an arbitrary executable's output
     *
     * @param executable path to the executable to run
     * @return Stream<String> of the executable's output
     */
    private Stream<String> getExecutableOutputAsLines(Path executable) {
        // read output of any arbitrary program into a string, then return it.
        ProcessBuilder pb = new ProcessBuilder(executable.toString());
        // run while setting CATALINA_HOME to the known correct value.
        if (SystemUtils.IS_OS_WINDOWS){
            pb.environment().put("CATALINA_HOME", Paths.get(idata.getInstallPath()).toString());
        }
        try {
            if (Files.exists(executable)) {
                Process exec = pb.start();
                exec.waitFor();
                BufferedReader br = new BufferedReader(new InputStreamReader(exec.getInputStream()));
                return br.lines();
            } else {
                Debug.log(String.format("%s executable does not exist.", executable.toString()));
                return Stream.empty();
            }
        } catch (InterruptedException e) {
            Debug.log(String.format("InterruptedException encountered when attempting to execute %s.", executable.toString()));
            return Stream.empty();
        } catch (IOException e) {
            Debug.log(String.format("IOException encountered when attempting to execute %s.", executable.toString()));
            return Stream.empty();
        }
    }

    /**
     * Processes the output of version.sh or version.bat to find the version of Tomcat the user is attempting to install into
     *
     * @param executableOutput Stream of version.sh or version.bat's output
     * @return String containing the value of the line starting with "Server number"
     */
    private String extractTomcatVersion(Stream<String> executableOutput) {
        return executableOutput.filter(line -> line.startsWith("Server number"))
                .map(line -> line.split(":"))
                .map(line -> line[1].trim())
                .findFirst().orElse("");
    }

    // EAP related methods
    private ContainerInformation isEapInstalled(Path installPath) {
        ContainerInformation eapInfo = new ContainerInformation(Container.EAP.toString());
        if (!containsAllPaths(installPath, requiredEapFiles)) {
            eapInfo.setState(InstallState.NotInstalled);
            return eapInfo;
        }
        if (containsAnyPath(installPath, eapProductFiles)) {
            eapInfo.setState(InstallState.ProductAlreadyInstalled);
            return eapInfo;
        }
        eapInfo.setVersion(new Version(getExistingEapVersionString(installPath)));
        VersionState versionCheck = isEapValidVersion(eapInfo.getVersion());
        switch (versionCheck) {
            case LowVersion:
                eapInfo.setState(InstallState.LowVersion);
                break;
            case HighVersion:
                eapInfo.setState(InstallState.HighVersion);
                break;
            case MissingVersion:
                eapInfo.setState(InstallState.MissingVersion);
                break;
            default:
                eapInfo.setState(InstallState.OK);
                break;
        }
        return eapInfo;
    }

    /**
     * Returns true if targetPath contains all paths in pathList, false if any are missing.
     * @param targetPath root to test for path existence
     * @param pathList list of paths to check for
     * @return true if all are present, false otherwise
     */
    private boolean containsAllPaths(Path targetPath, List<Path> pathList) {
        List<Path> missingFiles = findFiles(targetPath, pathList);
        return missingFiles.isEmpty();
    }

    /**
     * Returns true if targetPath contains any path in pathList, false if none are present.
     * @param targetPath root to test for path existence
     * @param pathList list of paths to check for
     * @return true if any are present, false otherwise.
     */
    private boolean containsAnyPath(Path targetPath, List<Path> pathList){
        List<Path> missingFiles = findFiles(targetPath, pathList);
        return !(missingFiles.size() == pathList.size());
    }

    private List<Path> findFiles(Path installPath, List<Path> files) {
        List<Path> missingFiles = new ArrayList<>();
        for (Path file : files) {
            Path filePath = installPath.resolve(file);
            if (!Files.exists(filePath)) {
                missingFiles.add(filePath);
            }
        }
        return missingFiles;
    }

    private VersionState isEapValidVersion(Version existingEapVersion) {
        if (existingEapVersion.getVersionString().isEmpty()) {
            return VersionState.MissingVersion;
        }
        Version maximumSupportedEapVersion = new Version(getMaximumSupportedEapVersion());
        Version minimumSupportedEapVersion = new Version(getSupportedEapVersion());

        if (existingEapVersion.compareTo(minimumSupportedEapVersion) < 0) {
            return VersionState.LowVersion;
        } else if (existingEapVersion.compareTo(maximumSupportedEapVersion) > 0) {
            return VersionState.HighVersion;
        } else {
            return VersionState.OK;
        }
    }

    private String getExistingEapVersionString(Path installPath) {
        String foundEapVersion = "";
        Path eapVersionPath = Paths.get(installPath.toString(), "version.txt");
        if (Files.exists(eapVersionPath)) {
            try (BufferedReader br = new BufferedReader(new FileReader(eapVersionPath.toString()))) {
                String versionLine = br.readLine();
                Pattern versionPattern = Pattern.compile(VERSION_REGEX);
                Matcher matcher = versionPattern.matcher(versionLine);
                if (matcher.find()) {
                    foundEapVersion = matcher.group();
                }
            } catch (IOException e) {
                Debug.log("IOException encountered when reading EAP version.txt");
            }
        }
        return foundEapVersion;
    }

    private String getMaximumSupportedEapVersion() {
        String majorVersion = idata.getVariable("eap.maximum.version.major");
        String minorVersion = idata.getVariable("eap.maximum.version.minor");
        int microRelease = Integer.parseInt(idata.getVariable("eap.maximum.version.micro"));
        if (microRelease > 9) {
            microRelease = microRelease % 10;
        }

        String releaseDesignation = idata.getVariable("eap.maximum.version.designation");
        return majorVersion + "." + minorVersion + "." + microRelease + "." + releaseDesignation;
    }

    private String getMaximumSupportedTomcatVersion() {
        return idata.getVariable("jws.maximum.version.major");
    }

    private String getSupportedEapVersion() {
        String majorVersion = idata.getVariable("eap.supported.version.major");
        String minorVersion = idata.getVariable("eap.supported.version.minor");
        String releaseVersion = idata.getVariable("eap.supported.version.micro");
        String minimumReleaseVersion = idata.getVariable("eap.supported.version.min.micro");
        String releaseDesignation = idata.getVariable("eap.supported.version.designation");
        if (minimumReleaseVersion == null) {
            return majorVersion + "." + minorVersion + "." + releaseVersion + "." + releaseDesignation;
        } else {
            return majorVersion + "." + minorVersion + "." + minimumReleaseVersion + "." + releaseDesignation;
        }
    }

    private void setRhbaProductFiles() {
        eapProductFiles.add(eapDeployments.resolve("business-central.war"));
        eapProductFiles.add(eapDeployments.resolve("kie-server.war"));
        tomcatProductFiles.add(webappsFolder.resolve("kie-server"));
        tomcatProductFiles.add(webappsFolder.resolve("controller"));
    }

    private void setRhdmProductFiles() {
        eapProductFiles.add(eapDeployments.resolve("decision-central.war"));
        eapProductFiles.add(eapDeployments.resolve("kie-server.war"));
        tomcatProductFiles.add(webappsFolder.resolve("kie-server"));
        tomcatProductFiles.add(webappsFolder.resolve("controller"));
    }

    private void setProductFileLists() {
        String productName = idata.getVariable("product.name");
        eapProductFiles = new ArrayList<>();
        tomcatProductFiles = new ArrayList<>();
        switch (productName) {
            case "rhba":
                setRhbaProductFiles();
                break;
            case "rhdm":
                setRhdmProductFiles();
                break;
        }
    }

    @Override
    public Status validateData(AutomatedInstallData adata) {
        this.idata = adata;
        setProductFileLists();
        Container discovered = getContainerInInstallationPath();
        switch (discovered) {
            case EAP:
                idata.setVariable("eap.install", "true");
                idata.setVariable("jws.install", "false");
                idata.setVariable("weblogic.install", "false");
                idata.setVariable("websphere.install", "false");
                return Status.OK;
            case Tomcat8:
                idata.setVariable("eap.install", "false");
                idata.setVariable("jws.install", "true");
                idata.setVariable("weblogic.install", "false");
                idata.setVariable("websphere.install", "false");
                return Status.OK;
            case Weblogic:
                idata.setVariable("eap.install", "true");
                idata.setVariable("jws.install", "false");
                idata.setVariable("weblogic.install", "true");
                idata.setVariable("websphere.install", "false");
                return Status.OK;
            case Websphere:
                idata.setVariable("eap.install", "false");
                idata.setVariable("jws.install", "false");
                idata.setVariable("weblogic.install", "false");
                idata.setVariable("websphere.install", "true");
                return Status.OK;
            default:
                idata.setVariable("eap.install", "false");
                idata.setVariable("jws.install", "false");
                idata.setVariable("weblogic.install", "false");
                idata.setVariable("websphere.install", "false");
                return Status.ERROR;
        }
    }

    @Override
    public String getErrorMessageId() {
        return error;
    }

    private void setErrorMessageId(String errorMessageId) {
        this.error = errorMessageId;
    }

    @Override
    public String getWarningMessageId() {
        return error;
    }


    public void setMessage(String messageId, String... format) {
        String messageText = idata.langpack.getString(messageId);
        this.message = String.format(messageText, format);
    }

    @Override
    public boolean getDefaultAnswer() {
        return true;
    }

    @Override
    public String getFormattedMessage() {
        return message;
    }


    class ContainerInformation {
        Version version;
        String name;
        InstallState state;

        ContainerInformation(String initialName) {
            setName(initialName);
        }

        void setState(InstallState newState) {
            state = newState;
        }

        InstallState getState() {
            return state;
        }

        void setName(String name) {
            this.name = name;
        }

        String getName() {
            return name;
        }

        void setVersion(Version newVersion) {
            version = newVersion;
        }

        Version getVersion() {
            return version;
        }
    }

    class Version implements Comparable<Version> {
        String version;

        Version(String version) {
            this.version = version;
        }

        String getVersionString() {
            return version;
        }

        public boolean equals(Object o) {
            if (this == o)
                return true;
            return o != null && this.getClass() == o.getClass() && this.compareTo((Version) o) == 0;
        }

        @Override
        public int compareTo(Version o) {
            if (o == null)
                return 1;
            String[] thisParts = this.getVersionString().split("(\\.)|(-)");
            String[] otherParts = o.getVersionString().split("(\\.)|(-)");
            int length = Math.min(thisParts.length, otherParts.length);
            for (int i = 0; i < (length == 1 ? 1 : length - 1); i++) {
                int thisPart = i < thisParts.length ? Integer.parseInt(thisParts[i]) : 0;
                int oPart = i < otherParts.length ? Integer.parseInt(otherParts[i]) : 0;
                if (thisPart < oPart)
                    return -1;
                if (thisPart > oPart)
                    return 1;
            }
            return 0;
        }
    }
}
