/*
 * IzPack - Copyright 2001-2009 Julien Ponge, All Rights Reserved.
 * 
 * Copyright 2009 Dennis Reil
 * 
 * http://izpack.org/
 * http://izpack.codehaus.org/
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 *     
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.izforge.izpack.panels;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.nio.file.InvalidPathException;
import java.util.List;
import java.util.stream.Collectors;

import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;

import com.izforge.izpack.gui.*;
import com.izforge.izpack.installer.InstallData;
import com.izforge.izpack.installer.InstallerFrame;
import com.izforge.izpack.installer.IzPanel;
import com.izforge.izpack.util.Debug;
import com.izforge.izpack.util.ErrorIzpackPathException;
import com.izforge.izpack.util.FileUtil;
import com.izforge.izpack.util.WarningIzpackPathException;

public class FileInputField extends JPanel implements ActionListener {

    private static final long serialVersionUID = 4673684743657328492L;

    InstallerFrame parentFrame;

    IzPanel parent;

    List<ValidatorContainer> validators;

    JTextField filetxt;

    JButton browseBtn;

    String set;

    int size;

    InstallData data;

    String fileExtension;

    String fileExtensionDescription;

    boolean allowEmpty;

    boolean userCancelled = false;

    boolean mustExist = true;

    protected static final int INVALID = 0, EMPTY = 1;
    private boolean suppressDownload = false;
    private int maxMessageWidth;

    public FileInputField(IzPanel parent, InstallData data, boolean directory, String set,
                          int size, List<ValidatorContainer> validatorConfig) {
        this(parent, data, directory, set, size, validatorConfig, null, null, true);
    }

    public FileInputField(IzPanel parent, InstallData data, boolean directory, String set,
                          int size, List<ValidatorContainer> validatorConfig, String fileExt, String fileExtDesc, boolean mustExist) {
        this.parent = parent;
        this.parentFrame = parent.getInstallerFrame();
        this.data = data;
        this.validators = validatorConfig;
        this.set = set;
        this.size = size;
        this.fileExtension = fileExt;
        this.fileExtensionDescription = fileExtDesc;
        this.mustExist = mustExist;
        this.suppressDownload = false;
        this.maxMessageWidth = data.guiPrefs.width - 175;
        this.initialize();
    }

    private void initialize() {
        filetxt = new JTextField(set, size);
        filetxt.setCaretPosition(0);
        parent.loadFileChooserLang();
        browseBtn = ButtonFactory.createButton(data.langpack.getString("UserInputPanel.search.browse"));
        browseBtn.addActionListener(this);
        browseBtn.addFocusListener(new FocusListenerAutoScroll());
        JFocusPanel focusBorder = new JFocusPanel(filetxt);
        filetxt.addFocusListener(new FocusListenerRepaint());
        filetxt.addFocusListener(new FocusListenerAutoScroll());
        this.add(focusBorder);
        this.add(browseBtn);
    }

    public void setFile(String filename) {
        filetxt.setText(filename);
    }

    public void actionPerformed(ActionEvent arg0) {
        if (arg0.getSource() == browseBtn) {
            Debug.trace("Show dirchooser");
            String initialPath = ".";
            if (filetxt.getText() != null) {
                initialPath = filetxt.getText();
            }
            JFileChooser filechooser = new JFileChooser(initialPath);
            prepareFileChooser(filechooser);

            if (filechooser.showOpenDialog(parentFrame) == JFileChooser.APPROVE_OPTION) {
                String selectedFile = filechooser.getSelectedFile().getAbsolutePath();
                filetxt.setText(selectedFile);
                Debug.trace("Setting current file chooser directory to: " + selectedFile);
            }
        }
    }

    protected void prepareFileChooser(JFileChooser filechooser) {
        filechooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
        if ((fileExtension != null) && (fileExtensionDescription != null)) {
            UserInputFileFilter fileFilter = new UserInputFileFilter();
            fileFilter.setFileExt(fileExtension);
            fileFilter.setFileExtDesc(fileExtensionDescription);
            filechooser.setFileFilter(fileFilter);
        }
    }

    public File getSelectedFile() {
        File result = null;
        if (filetxt.getText() != null) {
            result = new File(filetxt.getText());
        }
        return result;
    }

    protected void showMessage(int k) {
        if (k == INVALID) {
            showMessage("file.notfile");
        } else if (k == EMPTY) {
            showMessage("file.nofile");
        }
    }

    protected void showMessage(String messageType) {
        parent.emitWarning(parentFrame.langpack.getString("UserInputPanel." + messageType + ".caption"),
                parentFrame.langpack.getString("UserInputPanel." + messageType + ".message"));
    }

    protected boolean runValidators(String input){
        StringInputProcessingClient processingClient = new StringInputProcessingClient(
                input, validators);
        List<Validator.Status> success = processingClient.validate();
        if (success.contains(Validator.Status.ERROR)) {
            JOptionPane.showMessageDialog(parentFrame,
                    createJTextMessagePane(processingClient.getValidationMessage().get(0)),
                    parentFrame.langpack.getString("UserInputPanel.error.caption"),
                    JOptionPane.ERROR_MESSAGE);
        } else if (success.contains(Validator.Status.WARNING)){
            String warningMessages = processingClient.getValidationMessage().stream()
                    .collect(Collectors.joining("\n"));
            JOptionPane.showMessageDialog(parentFrame,
                    createJTextMessagePane(warningMessages),
                    parentFrame.langpack.getString("UserInputPanel.warning.caption"),
                    JOptionPane.WARNING_MESSAGE);
        }
        return !success.contains(Validator.Status.ERROR);
    }

    /**
     * Select a specific langpack string to put into the message dialog for FileInputField values that fail validation
     * @param messageLangpackId
     * @param format
     */
    private void showError(String messageLangpackId, String titleLangpackId, String... format){
        String message = parentFrame.langpack.getString(messageLangpackId);
        String title = parentFrame.langpack.getString(titleLangpackId);
        if (format.length == 0){
            JOptionPane.showMessageDialog(parentFrame, createJTextMessagePane(message), title, JOptionPane.ERROR_MESSAGE);
        } else {
            JOptionPane.showMessageDialog(parentFrame, createJTextMessagePane(String.format(message, format)), title, JOptionPane.ERROR_MESSAGE);
        }
    }

    private boolean showWarning(String messageLangpackId, String titleLangpackId, String... format){
        int response;
        String message = parentFrame.langpack.getString(messageLangpackId);
        String title = parentFrame.langpack.getString(titleLangpackId);
        if (format.length == 0){
            response = JOptionPane.showConfirmDialog(parentFrame, createJTextMessagePane(message), title, JOptionPane.YES_NO_OPTION);
        } else {
            response = JOptionPane.showConfirmDialog(parentFrame, createJTextMessagePane(String.format(message, format)), title, JOptionPane.YES_NO_OPTION);
        }
        return response == JOptionPane.YES_OPTION;
    }

    private JTextMessagePane createJTextMessagePane(String message){
        return new JTextMessagePane(message, maxMessageWidth, getFontMetrics(getFont()));
    }

    public boolean validateField() {
        boolean result = false;
        String input = filetxt.getText();

        if (allowEmpty && ((input == null) || (input.isEmpty()))) {
            result = true;
        } else if (input != null) {
            // Expand unix home reference
            if (input.startsWith("~")) {
                String home = System.getProperty("user.home");
                input = home + input.substring(1);
            }

            // Normalize the path
            File file = new File(input).getAbsoluteFile();
            input = file.toString();

            filetxt.setText(input);
            try {
                if (!FileUtil.isPathValid(input)) {
                    showMessage(INVALID);
                    return false;
                }
            } catch (WarningIzpackPathException wipe) {
                result = showWarning(wipe.getMessage(), "installer.warning");
                if (!result) return false;
            } catch (ErrorIzpackPathException ipe){
                showError(ipe.getMessage(),"installer.error");
                return false;
            }

            if (this.mustExist) {
                if (!_validate(file)) {
                    result = false;
                    if (!this.userCancelled)
                        showMessage(INVALID);
                } else {
                    // validation is successful
                    result = true;
                }
            } else {
                // File doesn't have to exist:
                result = runValidators(input);
            }
        } else {
            showMessage(EMPTY);
        }
        return result;
    }

    protected boolean _validate(File file) {
        return file.isFile();
    }

    public void setAllowEmptyInput(boolean allowEmpty) {
        this.allowEmpty = allowEmpty;
    }

    public void setMustExist(boolean mustExist) {
        this.mustExist = mustExist;
    }

    public boolean mustExist() {
        return this.mustExist;
    }

    public JTextField getField() {
        return this.filetxt;
    }

    public JButton getButton() {
        return this.browseBtn;
    }

    @Override
    public void setEnabled(boolean enabled) {
        filetxt.setEnabled(enabled);
        filetxt.setEditable(enabled);
        browseBtn.setEnabled(enabled);
    }

    @Override
    public boolean isEnabled(){
        return filetxt.isEnabled();
    }

    public void suppressDownload(boolean suppress) {
        this.suppressDownload = suppress;
    }

    public boolean isSuppressed(){
        return this.suppressDownload;
    }

    public String getInputText(){
        return filetxt.getText();
    }
}
