/*
 * Copyright (c) 2011-2014 The original author or authors
 * ------------------------------------------------------
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Apache License v2.0 which accompanies this distribution.
 *
 *     The Eclipse Public License is available at
 *     http://www.eclipse.org/legal/epl-v10.html
 *
 *     The Apache License v2.0 is available at
 *     http://www.opensource.org/licenses/apache2.0.php
 *
 * You may elect to redistribute this code under either of these licenses.
 */

package io.vertx.ext.jdbc;

import io.vertx.core.Vertx;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.ext.sql.SQLClient;
import io.vertx.ext.sql.SQLConnection;
import io.vertx.ext.sql.SQLOptions;
import io.vertx.ext.unit.Async;
import io.vertx.ext.unit.TestContext;
import io.vertx.ext.unit.junit.VertxUnitRunner;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

/**
 * @author <a href="mailto:pmlopes@gmail.com">Paulo Lopes</a>
 */
@RunWith(VertxUnitRunner.class)
public class JDBCTypesTestBase {

  private final Vertx vertx = Vertx.vertx();
  private SQLClient client;

  private static final List<String> SQL = new ArrayList<>();

  static {
    //TODO: Create table with more types for testing
    SQL.add("create table insert_table (id int not null primary key generated always as identity (START WITH 1, INCREMENT BY 1), lname varchar(255), fname varchar(255), dob date )");
    SQL.add("create table insert_tableNoIdentity (id int not null primary key, lname varchar(255), fname varchar(255), dob date )");
  }

  @Before
  public void setUp(TestContext should) throws SQLException {
    final Async test = should.async();

    JsonObject config = ConfigFactory.createConfigForDerby();
    client = JDBCClient.createNonShared(vertx, config);

    client.getConnection(res -> {
      should.assertTrue(res.succeeded());
      SQLConnection conn = res.result();
      // abuse the system here!
      // i'll run blocking code in order to quickly setup the test database
      Connection jdbcConn = conn.unwrap();
      try {
        for (String sql : SQL) {
          jdbcConn.createStatement().execute(sql);
        }
      } catch (SQLException e) {
        should.fail(e);
      }
      conn.close(res1 -> {
        should.assertTrue(res.succeeded());
        test.complete();
      });
    });

    test.await();
  }

  @After
  public void after(TestContext should) {
    final Async test = should.async();
    client.close(res -> {
      should.assertTrue(res.succeeded());
      test.complete();
    });
    test.await();
  }

  @Test
  public void testInsertWithNullParameters(TestContext should) {
    final Async test = should.async();

    client.getConnection(res -> {
      should.assertTrue(res.succeeded());
      SQLConnection conn = res.result();

      String sql = "INSERT INTO insert_table (lname, fname, dob) VALUES (?, ?, ?)";
      JsonArray params = new JsonArray().addNull().addNull().add("2002-02-02");
      conn
        .setOptions(new SQLOptions().setAutoGeneratedKeys(true))
        .updateWithParams(sql, params, res1 -> {
          should.assertTrue(res1.succeeded());
          should.assertNotNull(res1.result());
          should.assertEquals(1, res1.result().getUpdated());

          int id = res1.result().getKeys().getInteger(0);
          conn.queryWithParams("SElECT DOB FROM insert_table WHERE id=?", new JsonArray().add(id), res2 -> {
            should.assertTrue(res2.succeeded());
            should.assertNotNull(res2.result());
            should.assertEquals(1, res2.result().getResults().size());
            test.complete();
          });
        });
    });

    test.await();
  }

  /**
   * Test that insert and update works in a table without an identity column.
   */
  @Test
  public void testInsertUpdateNoIdentity(TestContext should) {
    final Async test = should.async();

    client.getConnection(res -> {
      should.assertTrue(res.succeeded());
      SQLConnection conn = res.result();
      String insertsql = "INSERT INTO insert_tableNoIdentity (id, lname, fname, dob) VALUES (?, ?, ?, ?)";
      JsonArray insertparams = new JsonArray().add(1).add("LastName1").addNull().add("2002-02-02");
      conn.updateWithParams(insertsql, insertparams, res1 -> {
        should.assertTrue(res1.succeeded());
        should.assertNotNull(res1.result());
        should.assertEquals(1, res1.result().getUpdated());

        should.assertTrue(res1.result().getKeys().isEmpty() || res1.result().getKeys().getValue(0) == null);
        int insertid = 1;

        conn.queryWithParams("SElECT lname FROM insert_tableNoIdentity WHERE id=?", new JsonArray().add(1), res2 -> {
          should.assertTrue(res2.succeeded());
          should.assertNotNull(res2.result());

          should.assertEquals(1, res2.result().getResults().size());
          should.assertEquals("LastName1", res2.result().getResults().get(0).getValue(0));

          // Now test that update works
          String updSql = "UPDATE insert_tableNoIdentity SET lname=? WHERE id=?";
          JsonArray updParams = new JsonArray().add("LastName2").add(insertid);

          conn.updateWithParams(updSql, updParams, res3 -> {
            should.assertTrue(res3.succeeded());
            should.assertNotNull(res3.result());

            should.assertEquals(1, res3.result().getUpdated());
            should.assertTrue(res3.result().getKeys().isEmpty() || res3.result().getKeys().getValue(0) == null);
            int updateid = 1;

            conn.queryWithParams("SElECT lname FROM insert_tableNoIdentity WHERE id=?", new JsonArray().add(updateid), res4 -> {
              should.assertTrue(res4.succeeded());
              should.assertNotNull(res4.result());

              should.assertNotNull(res4.result());
              should.assertEquals(1, res4.result().getResults().size());
              should.assertEquals("LastName2", res4.result().getResults().get(0).getValue(0));

              test.complete();
            });
          });
        });
      });
    });

    test.await();
  }
}
