#!/usr/bin/env bash

set -e

function usage {
    echo "Usage:"
    echo "   create-update.sh -p product_name -c component_list -v version -u updatable_versions -o output_directory -s maven_settings"
    echo ""
    echo "   -p, --product <string>             Product name: rhpam or rhdm."
    echo "   -c, --components <list>            Comma separated list of components to include in the change."
    echo "                                      I.e. business-central-eap7-deployable, kie-server-ee8, etc."
    echo "   -v, --version <version>            Current components version."
    echo "   -u, --updatableVersions <list>     List of updatable components versions sorted in ascending order."
    echo "   --latestRHDMVersion <version>      Latest supported RHDM version."
    echo "   -o, --outputDirectory <dir>        Output directory to place temporary and update change files."
    echo "   -w, --overwrittenPaths <list>      List of paths that will be forcedly included in the update."
    echo "   -s, --mavenSettings <file>         Path to Maven settings file."
}

while [[ $# -gt 0 ]]
do
    arg="$1"

    case "$arg" in
        -p|--product)
            shift
            PRODUCT="$1"
            ;;
        -c|--components)
            shift
            COMPONENTS="$1"
            ;;
        -v|--version)
            shift
            CURRENT_VERSION="$1"
            ;;
        -u|--updatableVersions)
            shift
            UPDATABLE_VERSIONS="$1"
            ;;
        --latestRHDMVersion)
            shift
            LATEST_RHDM_VERSION="$1"
            ;;
        -o|--outputDirectory)
            shift
            OUTPUT_DIR="$1"
            ;;
        -w|--overwrittenPaths)
            shift
            OVERWRITTEN_PATHS="$1"
            ;;
        -s|--mavenSettings)
            shift
            MAVEN_SETTINGS="$1"
            ;;
        *)
            echo "Unknown argument: $1"
            usage
            exit 1
            ;;
    esac
    shift
done

if [[ -z "$PRODUCT" || -z "$COMPONENTS" || -z "$CURRENT_VERSION" || -z "$UPDATABLE_VERSIONS" || -z "$OUTPUT_DIR"  || -z "$MAVEN_SETTINGS" ]];
then
    usage
    exit 0
fi

# Returns the minor version value
function get-version-minor() {
    echo $1 | awk -F"." '{ print $2 }'
}

PRODUCT=$(echo "${PRODUCT}" | tr '[:upper:]' '[:lower:]')
COMPONENTS=($(echo "${COMPONENTS}" | tr ',' '\n'))

UPDATABLE_VERSIONS=($(echo "${UPDATABLE_VERSIONS}" | tr ',' '\n'))
ALL_VERSIONS=("${UPDATABLE_VERSIONS[@]}" "${CURRENT_VERSION}")

OVERWRITTEN_PATHS=($(echo "${OVERWRITTEN_PATHS}" | tr ',' '\n'))

COMPONENTS_DIR="${OUTPUT_DIR}/components"
CHECKSUMS_DIR="${OUTPUT_DIR}/checksums"
CHANGES_DIR="${OUTPUT_DIR}/changes"
NEW_CONTENT_DIR="${CHANGES_DIR}/new-content"

LATEST_DM_MICRO_VERSION=$(get-version-minor ${LATEST_RHDM_VERSION})
DECISION_CENTRAL_COMPONENT="decision-central-eap7-deployable"

# Cleans up the output directory and recreates the directory tree to store working data and the update changes
function prepare-workspace {
    echo "Preparing workspace ..."
    rm -rf "${COMPONENTS_DIR}" "${CHECKSUMS_DIR}" "${CHANGES_DIR}"
    mkdir -p "${NEW_CONTENT_DIR}"
    for version in "${ALL_VERSIONS[@]}"
    do
        mkdir -p "${CHECKSUMS_DIR}/${version}"
    done
    for component in ${COMPONENTS[@]}
    do
        mkdir -p "${CHANGES_DIR}/${component}"
    done
}

# Downloads and unpack using Maven the component binaries to the output directory
function unpack-dependencies {
    for version in "${ALL_VERSIONS[@]}"
    do
        echo "Unpacking dependencies for version ${version} ..."
        for component in ${COMPONENTS[@]}
        do
            componentDir="${COMPONENTS_DIR}/${version}/${component}"
            local product_name=${PRODUCT}
            if [[ ${component} == ${DECISION_CENTRAL_COMPONENT} ]];
            then
              if [[ $(get-version-minor ${version}) -gt ${LATEST_DM_MICRO_VERSION} ]];
              # Only download DM when available
              then
                continue
              fi
              product_name="rhdm"
            fi
            mvn dependency:unpack -s "${MAVEN_SETTINGS}" -Dartifact="org.kie.rhba:${product_name}:${version}:zip:${component}" -DoutputDirectory="${componentDir}"
        done
    done
}

# Removes the EAP versioning from directories name
function remove-eap-versioning {
    echo "Stripping out EAP versioning from directories ..."
    find "${COMPONENTS_DIR}" -type d -name "jboss-eap-*" -printf "%p %h/jboss-eap\n" | xargs -rt -n2 mv
}

# Computes the MD5 checksums for every file in a component and version
function compute-checksums {
    for component in ${COMPONENTS[@]}
    do
        echo "Computing checksums for ${component} ..."
        for version in "${ALL_VERSIONS[@]}"
        do
            if [[ ${component} == ${DECISION_CENTRAL_COMPONENT} && $(get-version-minor ${version}) -gt ${LATEST_DM_MICRO_VERSION} ]];
            then
              # Use previous checksum file when DM is not available for current version
              cp "${CHECKSUMS_DIR}/${LATEST_RHDM_VERSION}/${component}.txt" "${CHECKSUMS_DIR}/${version}/${component}.txt"
              continue
            fi
            componentDir="${COMPONENTS_DIR}/${version}/${component}"
            for file in $(find ${componentDir} -type f -printf '%P\n' | sort)
            do
                checksum=$(md5sum "${componentDir}/${file}" | awk '{print $1}')
                echo "${file}=${checksum}" >> "${CHECKSUMS_DIR}/${version}/${component}.txt"
            done
        done
        cp "${CHECKSUMS_DIR}/${CURRENT_VERSION}/${component}.txt" "${CHANGES_DIR}/${component}/checksums.txt"
    done
}

# Aggregates the MD5 checksums by component for every version
function aggregate-checksums {
    for component in ${COMPONENTS[@]}
    do
        echo "Aggregating checksums for ${component} ..."
        aggregatedChecksums="${CHANGES_DIR}/${component}/checksums.txt"
        for version in "${ALL_VERSIONS[@]}"
        do
            prevVersionChecksums="${CHECKSUMS_DIR}/${version}/${component}.txt"
            cat "${prevVersionChecksums}" >> "${aggregatedChecksums}"
        done
        cat "${aggregatedChecksums}" | sort -u > "${aggregatedChecksums}.tmp"
        mv -f "${aggregatedChecksums}.tmp" "${aggregatedChecksums}"
    done
}

# Computes the new content per component as the difference between the $CURRENT_VERSION and every $UPDATABLE_VERSIONS
function compute-contents-to-add {
    for component in ${COMPONENTS[@]}
    do
        echo "Computing contents to add to ${component} ..."
        currentVersionChecksums="${CHECKSUMS_DIR}/${CURRENT_VERSION}/${component}.txt"
        newContent="${CHANGES_DIR}/${component}/new-content.txt"
        tempNewContent="${newContent}.tmp"
        for version in "${UPDATABLE_VERSIONS[@]}"
        do
            if [[ ${component} == ${DECISION_CENTRAL_COMPONENT} && $(get-version-minor ${version}) -ge ${LATEST_DM_MICRO_VERSION} ]];
            then
              # Compute only for existing DM versions
              continue
            fi
            prevVersionChecksums="${CHECKSUMS_DIR}/${version}/${component}.txt"
            grep -Fxv -f "${prevVersionChecksums}" "${currentVersionChecksums}" >> "${tempNewContent}"
        done
        for path in ${OVERWRITTEN_PATHS[@]}
        do
            grep -E "^${path}.*$" "${currentVersionChecksums}" >> "${tempNewContent}" || true
        done
        # format from "{path}={md5sum}" to "{path}={filename}.{md5sum}"
        sort -u "${tempNewContent}" | sed 's/\(.*\/\)\(.*\)=/\1\2=\2./g' > "${newContent}"
        rm -f "${tempNewContent}"
    done
}

# Computes the contents to be removed by component as the aggregation of all $UPDATABLE_VERSIONS minus the $CURRENT_VERSION
function compute-contents-to-remove {
    for component in ${COMPONENTS[@]}
    do
        echo "Computing contents to remove from ${component} ..."
        aggregatedChecksums="${CHANGES_DIR}/${component}/checksums.txt"
        currentVersionChecksums="${CHECKSUMS_DIR}/${CURRENT_VERSION}/${component}.txt"
        removeList="${CHANGES_DIR}/${component}/remove-list.txt"
        tempRemoveList="${removeList}.tmp"
        grep -Fxv -f "${currentVersionChecksums}" "${aggregatedChecksums}" > "${removeList}"
        awk -F '=' '{print $1}' "${removeList}" > "${tempRemoveList}"
        sed -E 's/-redhat-[0-9]+\.jar/*.jar/g' "${tempRemoveList}" | sort -u > "${removeList}"
        rm -f "${tempRemoveList}"
    done
}

# Copies all the components' new files to a common place to avoid repeating them per component
function aggregate-contents-to-add {
    for component in ${COMPONENTS[@]}
    do
        echo "Aggregating contents for ${component} ..."
        currentVersionComponentsDir="${COMPONENTS_DIR}/${CURRENT_VERSION}/${component}"
        componentNewContent="${CHANGES_DIR}/${component}/new-content.txt"
        while IFS= read -r newContent;
        do
            filePath="${newContent/=*/}"
            fileName="${newContent/*=/}"
            if [[ -f "${currentVersionComponentsDir}/${filePath}" && ! -f "${NEW_CONTENT_DIR}/${fileName}" ]]
            then
                cp "${currentVersionComponentsDir}/${filePath}" "${NEW_CONTENT_DIR}/${fileName}"
            fi
        done < "${componentNewContent}"
    done
}

prepare-workspace
unpack-dependencies
remove-eap-versioning
compute-checksums
aggregate-checksums
compute-contents-to-add
compute-contents-to-remove
aggregate-contents-to-add
