package org.jboss.brmsbpmsuite.patching.client;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import java.io.File;
import java.util.Arrays;
import java.util.List;

public class PatcherFactory {
    private static final Logger logger = LoggerFactory.getLogger(PatcherFactory.class);

    private static final Patcher NOOP_PATCHER = new Patcher() {
        @Override
        public void backup(File backupDir) {}

        @Override
        public void apply() {}
    };

    public static DistributionPatcher newDistributionPatcher(ClientPatcherConfig config) {
        DistributionType distributionType = config.getDistributionType();
        DistributionChecker distroChecker = DistributionCheckerFactory.create(distributionType, config.getProduct());
        VersionChecker versionChecker = VersionCheckerProvider.getInstance();
        List<Patcher> delegatePatchers = Arrays.asList(newDirectoryPatcher(config), newSecurityPolicyPatcher(config));
        return new GeneralDistributionPatcher(config.getDistributionRoot(), distroChecker, versionChecker, delegatePatchers);
    }

    public static Patcher newDirectoryPatcher(ClientPatcherConfig config) {
        PatchFilesReader reader = new PatchFilesReader(config.getPatchBasedir(), config.getDistributionType());
        String updateType = config.getDistributionType().getName();
        return new DirectoryPatcher(config.getDistributionRoot(), reader.removeListFor(updateType),  reader.protectedlist(),
                reader.newContentFor(updateType), reader.checksumsFor(updateType));
    }

    public static Patcher newSecurityPolicyPatcher(ClientPatcherConfig config) {
        File root = config.getDistributionRoot();
        switch (config.getDistributionType()) {
            case RHPAM_EAP7X_BC:
                return NOOP_PATCHER;
            case RHPAM_EE8_KIE: case RHPAM_EE7_CTRL:
                root = new File(config.getDistributionRoot(), "../../../bin");
        }

        if (!root.exists()) {
            logger.warn("Security policies directory couldn't be found: " + root.toString());
            return NOOP_PATCHER;
        }

        PatchFilesReader reader = new PatchFilesReader(config.getPatchBasedir(), config.getDistributionType());
        String updateType = config.getDistributionType().getName() + "-security-policy";
        return new DirectoryPatcher(root, reader.removeListFor(updateType),  reader.protectedlist(),
                reader.newContentFor(updateType), reader.checksumsFor(updateType));
    }

    public static Patcher newConfigPatcher(ClientPatcherConfig config) {
        if (config.getDistributionType() != DistributionType.RHPAM_EAP7X_BC) {
            return NOOP_PATCHER;
        }

        File patchesDir = new File(config.getPatchBasedir(), "updates/configuration");
        File standaloneDir = config.getDistributionRoot().getParentFile().getParentFile();
        File configDir = new File(standaloneDir, "configuration");
        try {
            return new ConfigurationPatcher(patchesDir, configDir);
        } catch (ParserConfigurationException | TransformerConfigurationException e) {
            throw new IllegalStateException("Could not instantiate the configuration patcher.", e);
        }
    }

}
