package com.redhat.installer.asconfiguration.utils;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.JarURLConnection;
import java.net.URL;
import java.nio.file.NoSuchFileException;
import java.util.Enumeration;
import java.util.Properties;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

public class JarUtils {

    public static String MANIFEST_PATH = "META-INF/MANIFEST.MF$";
    public static String POM_PATH = "META-INF/maven/.*/pom\\.properties$";

    /**
     * Returns an int based upon the result of various checks:<br/>
     * 0 : all is well <br/>
     * 1 : given path doesn't exist / is a directory<br/>
     * 2 : given path is not a zip<br/>
     * 3 : given path is an empty zip<br/>
     * 4 : given remote path is not accessible by the installer<br/>
     * 5 : Exception
     *
     * @param jar
     * @return
     */
    public static int verifyJarPath(String jar) {
        if (jar.equals("http://") || jar.equals("ftp://") | jar.equals(("https://")) || jar.equals("file://")) {
            return 4;
        } else if (jar.startsWith("http://") || jar.startsWith("https://")) {
            try {
                HttpURLConnection.setFollowRedirects(true);
                HttpURLConnection connection = (HttpURLConnection) new URL(jar).openConnection();
                connection.setRequestMethod("HEAD");
                int resp = connection.getResponseCode();
                if (resp != HttpURLConnection.HTTP_OK) {
                    return 4;
                }
            } catch (Exception e) {
                return 4;
            }
        } else if (jar.startsWith("ftp://") || jar.startsWith("file://")) {
            try {
                URL url = new URL(jar);
                InputStream check = url.openStream();
                check.close();
            } catch (Exception ex) {
                return 4;
            }
        } else {
            ZipFile zipFile = null;
            try {
                zipFile = new ZipFile(jar);
                Enumeration<? extends ZipEntry> entries = zipFile.entries();
                // we reached here and no entries? not a zip file.
                if (!entries.hasMoreElements()) {
                    // close the file
                    try {
                        zipFile.close();
                    } catch (IOException ioe) {
                        // more catastrophe
                        ioe.printStackTrace();
                    }
                    return 3;
                }
            // Java 8 throws FNFE, but Java 11+ throws NSFE
            } catch (FileNotFoundException | NoSuchFileException fnf) {
                // file doesn't exist.
                return 1;
            } catch (ZipException ze) {
                // file is not a zip
                return 2;
            } catch (IOException ioe) {
                // catastrophe
                ioe.printStackTrace();
                return 5;
            } finally {

                if (zipFile != null) {
                    try {
                        zipFile.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        return 0;
    }

    public static boolean checkValueOfResource(JarFile jar, String path, String attribute, String value) throws IOException {
        return checkValueOfResource(jar, path, attribute, value, false);
    }

    public static boolean checkValueOfResource(JarFile jar, String path, String attribute, String value, boolean exact) throws IOException {

        String valOfResource = getResourceFromJar(jar, path, attribute);

        if (!valOfResource.equals(""))
            return (exact ? (valOfResource.toLowerCase().equals(value.toLowerCase())) : valOfResource.toLowerCase().contains(value.toLowerCase()));
        return false;
    }

    public static JarFile getJarFromPath(String jarPath) throws IOException {

        JarFile jar;
        if (jarPath.startsWith("http://") || jarPath.startsWith("ftp://") || jarPath.startsWith("https://")
                || jarPath.startsWith("file://")){
            URL urlToJar = new URL(jarPath);
            URL url = new URL(urlToJar, "jar:" + urlToJar + "!/");
            JarURLConnection jarConnection = (JarURLConnection) url.openConnection();
            jar = jarConnection.getJarFile();
        } else {
            jar = new JarFile(jarPath);
        }
        return jar;
    }

    public static String getResourceFromJar(JarFile jar, String path, String attribute) throws IOException {

       JarEntry foundEntry = regexPath(jar, path);

        if (foundEntry != null) {
            Properties prop = new Properties();
            InputStream pom = jar.getInputStream(foundEntry);
            prop.load(pom);
            pom.close();
            String res = prop.getProperty(attribute);
            return res != null ? res.trim() : "";
        }
            return "";
    }

    private static JarEntry regexPath(JarFile jar, String path){
        Pattern regexPath =  Pattern.compile(path);
        Enumeration<JarEntry> entries = jar.entries();

        while(entries.hasMoreElements()){
            JarEntry entry = entries.nextElement();
            Matcher match = regexPath.matcher(entry.getName());
            if(match.matches()){
                return entry;
            }
        }
        return null;
    }
}
