# This script is a helper to set the productized versions of the dependencies when building the
# installer-commons and rhba-installer projects.
#
# The output of the script is a list of properties (-D<property_name>=<productized_version>) that
# can be appended to your local maven command and will set the version of the components to the
# productized version (something similar to what Repour does).
#
# This script was tested with Python 3 and it can be run with command:
# $ python installer_build_prod_dependencies.py
#
#
# Note: This script requires Bacon 2.1.11-SNAPSHOT version for now.
# The "bacon da" command is not implemented yet as part of the latest released version.
# To install the latest Bacon SNAPSHOT version use the command:
# $ curl -fsSL https://raw.github.com/project-ncl/bacon/main/bacon_install.py | python3 - snapshot
#
import os
import yaml

class Artifact:
    def __init__(self, gav, property_name):
        self.gav = gav
        self.property_name = property_name


def create_installer_commons_list():
    list = []
    list.append(Artifact("org.codehaus.izpack:izpack-standalone-compiler:4.5.4.rhba", "version.org.codehaus.izpack"))
    list.append(Artifact("org.jboss.eap:jboss-eap-parent:7.4.0.GA", "version.eap"))
    return list

def create_rhba_installer_list():
    list = []
    list.append(Artifact("org.kie.rhba:rhpam:7.12.0", "version.com.redhat.ba"))
    list.append(Artifact("org.kie:kie-bom:7.59.0.Final", "version.org.kie"))
    list.append(Artifact("org.jboss.eap:jboss-eap-parent:7.4.0.GA", "version.org.jboss.eap"))
    list.append(Artifact("org.jboss.installer:installer-commons:2.4.0.rhba", "version.org.jboss.installer"))
    list.append(Artifact("org.codehaus.izpack:izpack-standalone-compiler:4.5.4.rhba", "version.org.codehaus.izpack"))
    return list

def main():
    installer_commons_list = create_installer_commons_list()
    print("- Finding productized builds for installer commons")
    print_maven_parameters(installer_commons_list)

    rhba_installer_list = create_rhba_installer_list()
    print("\n- Finding productized builds for rhba installer")
    print_maven_parameters(rhba_installer_list)

def print_maven_parameters(list):
    mvn_parameteres = ""
    for i in list:
        prod_version = get_latest_prod_version(i.gav)
        print("Found produtized version {} for GAV {}".format(prod_version, i.gav))
        mvn_parameteres += "-D{}={} ".format(i.property_name, prod_version)
    print("maven parameters => ", mvn_parameteres)

def get_latest_prod_version(artifact_gav):
    cmd = "bacon da lookup maven {}".format(artifact_gav)
    stream = os.popen(cmd)
    cmd_output = stream.read()
    return yaml.safe_load(cmd_output)[0].get("bestMatchVersion")


if __name__ == "__main__":
    main()