package org.jboss.brmsbpmsuite.elytronmigration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

class FilesystemRealmTransformer {

    static class TransformationException extends RuntimeException {
        public TransformationException() {
            super();
        }

        public TransformationException(Throwable cause) {
            super(cause);
        }
    }

    private static final Logger LOGGER = LoggerFactory.getLogger(FilesystemRealmTransformer.class);

    private static final String KIE_FS_REALM_XML_RESOURCE = "/elytron/kie-fs-realm.xml" ;
    private static final String KIE_FS_REALM_TEMPLATE;
    static {
        LOGGER.info("Loading transformation template.");
        InputStream templateStream = ElytronMigrationApp.class.getResourceAsStream(KIE_FS_REALM_XML_RESOURCE);
        BufferedReader reader = new BufferedReader(new InputStreamReader(templateStream));
        KIE_FS_REALM_TEMPLATE = reader.lines().collect(Collectors.joining(System.lineSeparator()));
        if (KIE_FS_REALM_TEMPLATE.isEmpty()) {
            LOGGER.error("There was a problem loading the XSL transform template.");
            throw new TransformationException();
        }
    }

    private final Transformer transformer;
    private final Path root;

    FilesystemRealmTransformer(Path root) {
        this.root = root;
        this.transformer = initializeTransformer();
    }

    private Transformer initializeTransformer() {
        try {
            return TransformerFactory.newInstance().
                    newTransformer(new StreamSource(new StringReader(KIE_FS_REALM_TEMPLATE)));
        } catch (TransformerConfigurationException e) {
            LOGGER.error("There was a problem initializing the XML Transformer.");
            throw new TransformationException(e);
        }
    }

    void transform() {
        List<Path> paths;
        try (Stream<Path> rootWalk = Files.walk(root)) {
            paths = rootWalk.filter(Files::isRegularFile).
                    filter(path -> path.toString().toLowerCase().endsWith(".xml")).
                    collect(Collectors.toList());
        } catch (IOException e) {
            LOGGER.error("There was a problem retrieving the realm files.");
            throw new TransformationException(e);
        }
        if (paths.isEmpty()) {
            LOGGER.warn("No files to transform were detected.");
        }
        for (Path path : paths) {
            transformFile(path);
        }
    }

    private void transformFile(Path transformedPath) {
        LOGGER.info("Transforming file {}", transformedPath);
        Path tempFilePath;
        try {
            tempFilePath = Files.createTempFile(transformedPath.getFileName().toString(), ".tmp");
        } catch (IOException e) {
            LOGGER.error("There was a problem creating a temporary file.");
            throw new TransformationException(e);
        }
        try {
            transformer.transform(new StreamSource(transformedPath.toFile()), new StreamResult(tempFilePath.toFile()));
        } catch (TransformerException e) {
            LOGGER.error("There was a problem transforming the file {}.", transformedPath);
            throw new TransformationException(e);
        }
        try {
            Files.copy(tempFilePath, transformedPath, StandardCopyOption.REPLACE_EXISTING);
        } catch (IOException e) {
            LOGGER.error("There was a problem overwriting the original file {} with the transformed version {}.", transformedPath, tempFilePath);
            throw new TransformationException(e);
        }
    }
}
