package org.jboss.brmsbpmsuite.patching.client;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class DistributionCheckerFactory {

    /**
     * Creates specific distribution checker based on provided distribution type and target product.
     */
    public static DistributionChecker create(DistributionType distributionType, TargetProduct product) {
        List<ExpectedDistributionEntry> expectedEntries = getExpectedDistributionEntries(distributionType, product);
        switch (distributionType) {
            // in case there are specific checkers for other distributions, they should be created here
            default:
                return new ExpectedContentDistributionChecker(expectedEntries);
        }
    }

    private static List<ExpectedDistributionEntry> getExpectedDistributionEntries(DistributionType distributionType,
            TargetProduct product) {
        switch (distributionType) {

            // Business Central WARs
            case RHPAM_EAP7X_BC:
                return eapBCEntries(product);

            // KIE Server
            case RHPAM_EE8_KIE:
                return eeKieServerEntries();

            // Controller
            case RHPAM_EE7_CTRL:
                return eeControllerEntries();

            case RHPAM_JWS_KIE:
            case RHPAM_JWS_CTRL:
                return genericWebAppEntries();

            default:
                // this should not happen, the code above should handle all the distribution types
                throw new IllegalArgumentException("Unknown distribution type " + distributionType);
        }
    }

    private static List<ExpectedDistributionEntry> commonBCEntries(TargetProduct product) {
        ImmutableList.Builder<ExpectedDistributionEntry> listBuilder = new ImmutableList.Builder<ExpectedDistributionEntry>();
        listBuilder.add(new ExpectedExistingFile("WEB-INF/web.xml"));
        List<String> validValues;
        if (product == TargetProduct.RHPAM) {
            validValues = Arrays.asList("KIE Workbench - Distribution Wars", "Business Central - Distribution Wars", "WildFly: Bootable Jar boot");
        } else {
            throw new IllegalStateException(
                    "Product " + product + " can not be used to determine expected Business Central entries!");
        }
        listBuilder.add(new ExpectedManifestAttribute("Implementation-Title", validValues));
        return listBuilder.build();
    }

    private static List<ExpectedDistributionEntry> eapBCEntries(TargetProduct product) {
        return expectedEntriesList(
                commonBCEntries(product),
                new ExpectedExistingFile("WEB-INF/jboss-web.xml")
        );
    }

    private static List<ExpectedDistributionEntry> genericBCEntries(TargetProduct product) {
        return expectedEntriesList(
                commonBCEntries(product)
        );
    }

    private static List<ExpectedDistributionEntry> genericWebAppEntries() {
        return expectedEntriesList(
                new ExpectedExistingDir("WEB-INF"),
                new ExpectedExistingFile("WEB-INF/web.xml")
        );
    }

    private static List<ExpectedDistributionEntry> eeKieServerEntries() {
        return expectedEntriesList(
                genericWebAppEntries(),
                new ExpectedExistingFile("META-INF/kie-server-jms.xml")
        );
    }

    private static List<ExpectedDistributionEntry> eeControllerEntries() {
        return expectedEntriesList(
                genericWebAppEntries()
                //new ExpectedExistingDir("META-INF/services")
        );
    }

    private static List<ExpectedDistributionEntry> addPathPrefix(List<ExpectedDistributionEntry> expectedEntries,
            final String pathPrefix) {
        return Lists.transform(expectedEntries, new Function<ExpectedDistributionEntry, ExpectedDistributionEntry>() {
            @Override
            public ExpectedDistributionEntry apply(ExpectedDistributionEntry entry) {
                return entry.withPath(pathPrefix + entry.getPath());
            }
        });
    }

    private static <T extends ExpectedDistributionEntry> List<ExpectedDistributionEntry> expectedEntriesList(
            List<ExpectedDistributionEntry> list, T... entries) {
        List<ExpectedDistributionEntry> result = new ArrayList<ExpectedDistributionEntry>(Arrays.asList(entries));
        result.addAll(list);
        return result;
    }

    private static <T extends ExpectedDistributionEntry> List<ExpectedDistributionEntry> expectedEntriesList(T... entries) {
        return new ArrayList<ExpectedDistributionEntry>(Arrays.asList(entries));
    }

}
