/*
 * Copyright 2018 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.kie.workbench.common.dmn.client.editors.types.listview;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.google.gwtmockito.GwtMockitoTestRunner;
import elemental2.dom.HTMLElement;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.kie.workbench.common.dmn.api.definition.v1_1.ItemDefinition;
import org.kie.workbench.common.dmn.client.editors.types.common.DataType;
import org.kie.workbench.common.dmn.client.editors.types.common.DataTypeManager;
import org.kie.workbench.common.dmn.client.editors.types.listview.common.SmallSwitchComponent;
import org.kie.workbench.common.dmn.client.editors.types.listview.confirmation.DataTypeConfirmation;
import org.kie.workbench.common.dmn.client.editors.types.persistence.ItemDefinitionStore;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.InOrder;
import org.mockito.Mock;
import org.uberfire.mvp.Command;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.junit.Assert.assertEquals;
import static org.kie.workbench.common.dmn.client.editors.types.persistence.CreationType.ABOVE;
import static org.kie.workbench.common.dmn.client.editors.types.persistence.CreationType.BELOW;
import static org.kie.workbench.common.dmn.client.editors.types.persistence.CreationType.NESTED;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyBoolean;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.inOrder;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@RunWith(GwtMockitoTestRunner.class)
public class DataTypeListItemTest {

    @Mock
    private DataTypeListItem.View view;

    @Mock
    private DataTypeSelect dataTypeSelectComponent;

    @Mock
    private DataTypeConstraint dataTypeConstraintComponent;

    @Mock
    private SmallSwitchComponent dataTypeCollectionComponent;

    @Mock
    private DataType dataType;

    @Mock
    private ItemDefinitionStore itemDefinitionStore;

    @Captor
    private ArgumentCaptor<DataType> dataTypeCaptor;

    @Mock
    private DataTypeList dataTypeList;

    @Mock
    private DataTypeConfirmation confirmation;

    private DataTypeManager dataTypeManager;

    private DataTypeListItem listItem;

    @Before
    public void setup() {
        dataTypeManager = spy(new DataTypeManager(null, null, itemDefinitionStore, null, null, null, null, null));
        listItem = spy(new DataTypeListItem(view, dataTypeSelectComponent, dataTypeConstraintComponent, dataTypeCollectionComponent, dataTypeManager, confirmation));
        listItem.init(dataTypeList);
    }

    @Test
    public void testSetup() {
        listItem.setup();

        verify(view).init(listItem);
    }

    @Test
    public void testGetElement() {

        final HTMLElement expectedElement = mock(HTMLElement.class);

        when(view.getElement()).thenReturn(expectedElement);

        HTMLElement actualElement = listItem.getElement();

        assertEquals(expectedElement, actualElement);
    }

    @Test
    public void testSetupDataType() {

        final DataType expectedDataType = this.dataType;
        final int expectedLevel = 1;

        listItem.setupDataType(expectedDataType, expectedLevel);

        final InOrder inOrder = inOrder(listItem);
        inOrder.verify(listItem).setupSelectComponent();
        inOrder.verify(listItem).setupConstraintComponent();
        inOrder.verify(listItem).setupCollectionComponent();
        inOrder.verify(listItem).setupView();

        assertEquals(expectedDataType, listItem.getDataType());
        assertEquals(expectedLevel, listItem.getLevel());
    }

    @Test
    public void testSetupConstraintComponent() {

        final DataType dataType = mock(DataType.class);
        doReturn(dataType).when(listItem).getDataType();

        listItem.setupConstraintComponent();

        verify(dataTypeConstraintComponent).init(dataType);
    }

    @Test
    public void testSetupCollectionComponentWhenDataTypeIsCollection() {

        final DataType dataType = mock(DataType.class);
        final boolean isCollection = true;

        when(dataType.isCollection()).thenReturn(isCollection);
        doReturn(dataType).when(listItem).getDataType();

        listItem.setupCollectionComponent();

        verify(dataTypeCollectionComponent).setValue(isCollection);
        verify(listItem).refreshCollectionYesLabel();
    }

    @Test
    public void testSetupCollectionComponentWhenDataTypeIsNotCollection() {

        final DataType dataType = mock(DataType.class);
        final boolean isCollection = false;

        when(dataType.isCollection()).thenReturn(isCollection);
        doReturn(dataType).when(listItem).getDataType();

        listItem.setupCollectionComponent();

        verify(dataTypeCollectionComponent).setValue(isCollection);
        verify(listItem).refreshCollectionYesLabel();
    }

    @Test
    public void testSetupSelectComponent() {

        final DataType dataType = mock(DataType.class);
        doReturn(dataType).when(listItem).getDataType();

        listItem.setupSelectComponent();

        verify(dataTypeSelectComponent).init(listItem, dataType);
    }

    @Test
    public void testSetupView() {

        final DataType dataType = mock(DataType.class);
        when(listItem.getDataType()).thenReturn(dataType);

        listItem.setupView();

        verify(view).setupSelectComponent(dataTypeSelectComponent);
        verify(view).setupConstraintComponent(dataTypeConstraintComponent);
        verify(view).setupCollectionComponent(dataTypeCollectionComponent);
        verify(view).setDataType(dataType);
    }

    @Test
    public void testExpandOrCollapseSubTypesWhenViewIsCollapsed() {

        when(view.isCollapsed()).thenReturn(true);

        listItem.expandOrCollapseSubTypes();

        verify(listItem).expand();
    }

    @Test
    public void testExpandOrCollapseSubTypesWhenViewIsNotCollapsed() {

        when(view.isCollapsed()).thenReturn(false);

        listItem.expandOrCollapseSubTypes();

        verify(listItem).collapse();
    }

    @Test
    public void testCollapse() {

        listItem.collapse();

        verify(view).collapse();
    }

    @Test
    public void testExpand() {

        listItem.expand();

        verify(view).expand();
    }

    @Test
    public void testRefreshSubItems() {

        final DataType dataType = mock(DataType.class);
        final List<DataType> dataTypes = singletonList(dataType);

        listItem.refreshSubItems(dataTypes);

        verify(dataTypeList).refreshSubItemsFromListItem(listItem, dataTypes);
        verify(view).enableFocusMode();
        verify(view).toggleArrow(anyBoolean());
    }

    @Test
    public void testEnableEditMode() {

        final DataType dataType = mock(DataType.class);
        final String expectedName = "name";
        final String expectedType = "type";
        final String expectedConstraint = "constraint";
        final boolean expectedIsCollection = true;

        doReturn(dataType).when(listItem).getDataType();
        when(dataType.getName()).thenReturn(expectedName);
        when(dataType.getType()).thenReturn(expectedType);
        when(dataType.getConstraint()).thenReturn(expectedConstraint);
        when(dataType.isCollection()).thenReturn(expectedIsCollection);

        listItem.enableEditMode();

        assertEquals(expectedName, listItem.getOldName());
        assertEquals(expectedType, listItem.getOldType());
        assertEquals(expectedConstraint, listItem.getOldConstraint());
        assertEquals(expectedIsCollection, listItem.getOldIsCollection());

        verify(view).showSaveButton();
        verify(view).showDataTypeNameInput();
        verify(view).enableFocusMode();
        verify(view).showConstraintContainer();
        verify(view).hideConstraintText();
        verify(view).hideCollectionYesLabel();
        verify(view).showCollectionContainer();
        verify(dataTypeSelectComponent).enableEditMode();
        verify(dataTypeConstraintComponent).refreshView();
    }

    @Test
    public void testDisableEditMode() {

        doNothing().when(listItem).discardNewDataType();
        doNothing().when(listItem).closeEditMode();

        listItem.disableEditMode();

        verify(listItem).discardNewDataType();
        verify(listItem).closeEditMode();
    }

    @Test
    public void testSaveAndCloseEditModeWhenDataTypeIsValid() {

        final DataType dataType = spy(makeDataType());
        final DataType updatedDataType = spy(makeDataType());
        final Command doSaveAndCloseCommand = mock(Command.class);
        final Command doDisableEditMode = mock(Command.class);

        doReturn(dataType).when(listItem).getDataType();
        doReturn(updatedDataType).when(listItem).updateProperties(dataType);
        doReturn(true).when(updatedDataType).isValid();
        doReturn(doSaveAndCloseCommand).when(listItem).doSaveAndCloseEditMode(updatedDataType);
        doReturn(doDisableEditMode).when(listItem).doDisableEditMode();

        listItem.saveAndCloseEditMode();

        verify(confirmation).ifDataTypeDoesNotHaveLostSubDataTypes(updatedDataType, doSaveAndCloseCommand, doDisableEditMode);
    }

    @Test
    public void testDoDisableEditMode() {
        doNothing().when(listItem).disableEditMode();

        listItem.doDisableEditMode().execute();

        verify(listItem).disableEditMode();
    }

    @Test
    public void testSaveAndCloseEditModeWhenDataTypeIsNotValid() {

        final DataType dataType = spy(makeDataType());
        final DataType updatedDataType = spy(makeDataType());

        doReturn(dataType).when(listItem).getDataType();
        doReturn(updatedDataType).when(listItem).updateProperties(dataType);
        doReturn(false).when(updatedDataType).isValid();

        listItem.saveAndCloseEditMode();

        verify(confirmation, never()).ifDataTypeDoesNotHaveLostSubDataTypes(any(), any(), any());
    }

    @Test
    public void testDoSaveAndCloseEditMode() {

        final DataType dataType = spy(makeDataType());
        final List<DataType> updatedDataTypes = singletonList(makeDataType());

        doReturn(updatedDataTypes).when(listItem).persist(dataType);
        doReturn(dataType).when(listItem).getDataType();

        listItem.doSaveAndCloseEditMode(dataType).execute();

        verify(dataTypeList).refreshItemsByUpdatedDataTypes(updatedDataTypes);
        verify(listItem).closeEditMode();
    }

    @Test
    public void testPersist() {

        final String uuid = "uuid";
        final DataType dataType = spy(makeDataType());
        final ItemDefinition itemDefinition = mock(ItemDefinition.class);
        final List<DataType> subDataTypes = singletonList(makeDataType());
        final List<DataType> affectedDataTypes = emptyList();

        when(itemDefinitionStore.get(uuid)).thenReturn(itemDefinition);
        when(dataTypeSelectComponent.getSubDataTypes()).thenReturn(subDataTypes);
        doReturn(uuid).when(dataType).getUUID();
        doReturn(affectedDataTypes).when(dataType).update();

        listItem.persist(dataType);

        final InOrder inOrder = inOrder(dataTypeManager, dataType);

        inOrder.verify(dataTypeManager).from(dataType);
        inOrder.verify(dataTypeManager).withSubDataTypes(subDataTypes);
        inOrder.verify(dataTypeManager).get();
        inOrder.verify(dataType).update();
    }

    @Test
    public void testDiscardNewDataType() {

        final DataType dataType = spy(makeDataType());
        final List<DataType> subDataTypes = Collections.emptyList();
        final String expectedName = "name";
        final String expectedType = "type";
        final String expectedConstraint = "constraint";
        final boolean expectedIsCollection = true;

        doReturn(subDataTypes).when(dataType).getSubDataTypes();
        doReturn(dataType).when(listItem).getDataType();
        doReturn(expectedName).when(listItem).getOldName();
        doReturn(expectedType).when(listItem).getOldType();
        doReturn(expectedConstraint).when(listItem).getOldConstraint();
        doReturn(expectedIsCollection).when(listItem).getOldIsCollection();

        listItem.discardNewDataType();

        verify(view).setDataType(dataTypeCaptor.capture());
        verify(listItem).setupSelectComponent();
        verify(listItem).setupCollectionComponent();
        verify(listItem).refreshSubItems(subDataTypes);

        final DataType dataTypeCaptorValue = dataTypeCaptor.getValue();

        assertEquals(expectedName, dataTypeCaptorValue.getName());
        assertEquals(expectedType, dataTypeCaptorValue.getType());
        assertEquals(expectedConstraint, dataTypeCaptorValue.getConstraint());
        assertEquals(expectedIsCollection, dataTypeCaptorValue.isCollection());
    }

    @Test
    public void testCloseEditMode() {

        doReturn(dataType).when(listItem).getDataType();

        listItem.closeEditMode();

        verify(view).showEditButton();
        verify(view).hideDataTypeNameInput();
        verify(view).disableFocusMode();
        verify(view).hideConstraintContainer();
        verify(view).showConstraintText();
        verify(view).hideCollectionContainer();
        verify(listItem).refreshCollectionYesLabel();
        verify(dataTypeSelectComponent).disableEditMode();
    }

    @Test
    public void testRefreshCollectionYesLabelWhenDataTypeIsCollection() {

        doReturn(dataType).when(listItem).getDataType();
        when(dataType.isCollection()).thenReturn(true);

        listItem.refreshCollectionYesLabel();

        verify(view).showCollectionYesLabel();
    }

    @Test
    public void testRefreshCollectionYesLabelWhenDataTypeIsNotCollection() {

        doReturn(dataType).when(listItem).getDataType();
        when(dataType.isCollection()).thenReturn(false);

        listItem.refreshCollectionYesLabel();

        verify(view).hideCollectionYesLabel();
    }

    @Test
    public void testUpdateProperties() {

        final DataType dataType = spy(makeDataType());
        final String uuid = "uuid";
        final String expectedName = "name";
        final String expectedType = "type";
        final String expectedConstraint = "constraint";
        final boolean expectedCollection = true;
        final ItemDefinition itemDefinition = mock(ItemDefinition.class);

        when(dataType.getUUID()).thenReturn(uuid);
        when(itemDefinitionStore.get(uuid)).thenReturn(itemDefinition);
        when(view.getName()).thenReturn(expectedName);
        when(dataTypeSelectComponent.getValue()).thenReturn(expectedType);
        when(dataTypeConstraintComponent.getValue()).thenReturn(expectedConstraint);
        when(dataTypeCollectionComponent.getValue()).thenReturn(expectedCollection);
        when(dataTypeManager.get()).thenReturn(dataType);

        final DataType updatedDataType = listItem.updateProperties(dataType);

        assertEquals(expectedName, updatedDataType.getName());
        assertEquals(expectedType, updatedDataType.getType());
        assertEquals(expectedConstraint, updatedDataType.getConstraint());
        assertEquals(expectedCollection, updatedDataType.isCollection());
    }

    @Test
    public void testRefresh() {

        final DataType dataType = spy(makeDataType());
        final String expectedConstraint = "constraint";
        final String expectedName = "name";

        doReturn(expectedConstraint).when(dataType).getConstraint();
        doReturn(expectedName).when(dataType).getName();
        doReturn(dataType).when(listItem).getDataType();

        listItem.refresh();

        verify(dataTypeSelectComponent).refresh();
        verify(dataTypeSelectComponent).init(listItem, dataType);
        verify(view).setName(expectedName);
        verify(view).setConstraint(expectedConstraint);
        verify(listItem).setupCollectionComponent();
        verify(listItem).setupConstraintComponent();
    }

    @Test
    public void testRemove() {

        final DataType dataType = mock(DataType.class);
        final Command command = mock(Command.class);

        doReturn(dataType).when(listItem).getDataType();
        doReturn(command).when(listItem).doRemove();

        listItem.remove();

        verify(confirmation).ifIsNotReferencedDataType(dataType, command);
    }

    @Test
    public void testDoRemove() {

        final DataType dataType = mock(DataType.class);
        final DataType dataType0 = mock(DataType.class);
        final DataType dataType1 = mock(DataType.class);
        final DataType dataType2 = mock(DataType.class);
        final DataType dataType3 = mock(DataType.class);
        final List<DataType> destroyedDataTypes = new ArrayList<>(asList(dataType0, dataType1, dataType2, dataType3));
        final List<DataType> removedDataTypes = asList(dataType1, dataType2);

        doReturn(destroyedDataTypes).when(dataType).destroy();
        doReturn(removedDataTypes).when(listItem).removeTopLevelDataTypes(destroyedDataTypes);
        doReturn(dataType).when(listItem).getDataType();

        listItem.doRemove().execute();

        verify(dataTypeList).refreshItemsByUpdatedDataTypes(asList(dataType0, dataType3));
    }

    @Test
    public void testRemoveTopLevelDataTypes() {

        final DataType dataType = mock(DataType.class);
        final DataType dataType0 = mock(DataType.class);
        final DataType dataType1 = mock(DataType.class);
        final DataType dataType2 = mock(DataType.class);
        final DataType dataType3 = mock(DataType.class);

        when(listItem.getDataType()).thenReturn(dataType);
        when(dataType.getName()).thenReturn("tPerson");
        when(dataType0.getName()).thenReturn("tPerson");
        when(dataType0.isTopLevel()).thenReturn(true);
        when(dataType1.getType()).thenReturn("tPerson");
        when(dataType1.isTopLevel()).thenReturn(false);
        when(dataType2.getType()).thenReturn("tCity");
        when(dataType2.isTopLevel()).thenReturn(true);
        when(dataType3.getName()).thenReturn("tCity");
        when(dataType3.isTopLevel()).thenReturn(false);

        final List<DataType> actualDataTypes = listItem.removeTopLevelDataTypes(asList(dataType0, dataType1, dataType2, dataType3));
        final List<DataType> expectedDataTypes = singletonList(dataType0);

        verify(dataTypeList).removeItem(dataType0);
        assertEquals(expectedDataTypes, actualDataTypes);
    }

    @Test
    public void testInsertFieldAboveWhenTheNewDataTypeIsTopLevel() {

        final DataType newDataType = mock(DataType.class);
        final DataType reference = mock(DataType.class);
        final List<DataType> updatedDataTypes = asList(mock(DataType.class), mock(DataType.class));

        when(newDataType.isTopLevel()).thenReturn(true);
        when(newDataType.create(reference, ABOVE)).thenReturn(updatedDataTypes);
        doReturn(dataTypeManager).when(dataTypeManager).fromNew();
        doReturn(newDataType).when(dataTypeManager).get();
        doReturn(reference).when(listItem).getDataType();

        listItem.insertFieldAbove();

        verify(listItem).closeEditMode();
        verify(dataTypeList).insertAbove(newDataType, reference);
    }

    @Test
    public void testInsertFieldAboveWhenTheNewDataTypeIsNotTopLevel() {

        final DataType newDataType = mock(DataType.class);
        final DataType reference = mock(DataType.class);
        final List<DataType> updatedDataTypes = asList(mock(DataType.class), mock(DataType.class));

        when(newDataType.isTopLevel()).thenReturn(false);
        when(newDataType.create(reference, ABOVE)).thenReturn(updatedDataTypes);
        doReturn(dataTypeManager).when(dataTypeManager).fromNew();
        doReturn(newDataType).when(dataTypeManager).get();
        doReturn(reference).when(listItem).getDataType();

        listItem.insertFieldAbove();

        verify(listItem).closeEditMode();
        verify(dataTypeList).refreshItemsByUpdatedDataTypes(updatedDataTypes);
    }

    @Test
    public void testInsertFieldBelowWhenTheNewDataTypeIsTopLevel() {

        final DataType newDataType = mock(DataType.class);
        final DataType reference = mock(DataType.class);
        final List<DataType> updatedDataTypes = asList(mock(DataType.class), mock(DataType.class));

        when(newDataType.isTopLevel()).thenReturn(true);
        when(newDataType.create(reference, BELOW)).thenReturn(updatedDataTypes);
        doReturn(dataTypeManager).when(dataTypeManager).fromNew();
        doReturn(newDataType).when(dataTypeManager).get();
        doReturn(reference).when(listItem).getDataType();

        listItem.insertFieldBelow();

        verify(listItem).closeEditMode();
        verify(dataTypeList).insertBelow(newDataType, reference);
    }

    @Test
    public void testInsertFieldBelowWhenTheNewDataTypeIsNotTopLevel() {

        final DataType newDataType = mock(DataType.class);
        final DataType reference = mock(DataType.class);
        final List<DataType> updatedDataTypes = asList(mock(DataType.class), mock(DataType.class));

        when(newDataType.isTopLevel()).thenReturn(false);
        when(newDataType.create(reference, BELOW)).thenReturn(updatedDataTypes);
        doReturn(dataTypeManager).when(dataTypeManager).fromNew();
        doReturn(newDataType).when(dataTypeManager).get();
        doReturn(reference).when(listItem).getDataType();

        listItem.insertFieldBelow();

        verify(listItem).closeEditMode();
        verify(dataTypeList).refreshItemsByUpdatedDataTypes(updatedDataTypes);
    }

    @Test
    public void testInsertNestedField() {

        final DataType newDataType = mock(DataType.class);
        final DataType reference = mock(DataType.class);
        final List<DataType> updatedDataTypes = asList(mock(DataType.class), mock(DataType.class));

        when(newDataType.create(reference, NESTED)).thenReturn(updatedDataTypes);
        doReturn(dataTypeManager).when(dataTypeManager).fromNew();
        doReturn(newDataType).when(dataTypeManager).get();
        doReturn(reference).when(listItem).getDataType();

        listItem.insertNestedField();

        verify(dataTypeList).refreshItemsByUpdatedDataTypes(updatedDataTypes);
    }

    private DataType makeDataType() {
        return new DataType(null);
    }
}
