package com.izforge.izpack.util;

import com.izforge.izpack.installer.InstallData;
import org.aesh.readline.ConsoleBuffer;
import org.aesh.readline.InputProcessor;
import org.aesh.readline.Prompt;
import org.aesh.readline.Readline;
import org.aesh.readline.ReadlineBuilder;
import org.aesh.readline.action.Action;
import org.aesh.readline.completion.CompleteOperation;
import org.aesh.readline.completion.Completion;
import org.aesh.readline.editing.EditMode;
import org.aesh.readline.editing.EditModeBuilder;
import org.aesh.readline.terminal.Key;
import org.aesh.readline.tty.terminal.TerminalConnection;
import org.aesh.terminal.Attributes;
import org.aesh.terminal.tty.Signal;
import org.aesh.utils.Config;
import org.aesh.utils.OSUtils;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * AeshReadlineConsole.java replacement using AEsh readline
 * Created by thauser on 03/05/18.
 */
public class AeshReadlineConsole {
    private static TerminalConnection connection;
    private static Consumer<Void> defaultCloseHandler;

    public static String readLine() {
        return readConsole(new Prompt(""), Collections.emptyList());
    }

    public static String readPassword() {
        return readConsole(new Prompt("", '*'), Collections.emptyList());
    }

    public static char readChar() {
        String line = readLine();
        return line.toLowerCase().charAt(0);
    }

    public static String readFile(boolean isOnlyDirectories) {
        String fileLocation = readConsole(new Prompt(""), new ArrayList<Completion>(){{
            add(new FilenameCompletion(isOnlyDirectories));
        }});
        return fileLocation.replaceFirst("^~", System.getProperty("user.home"));
    }

    /**
     * Need to be able to interrupt this reading with a keypress, for the Console Download display.
     * It may be that with aesh readline, a simple readChar() is enough. identical to readLine() at the moment
     * @return
     */

    public static String readLineInterruptable() throws InterruptedException {
        return readLine();
    }

    private static String readConsole(Prompt prompt, List<Completion> completions) {
        String[] out = new String[1];
        try {
            initializeConnection();
            Readline readline = ReadlineBuilder.builder().enableHistory(false).editMode(createIzpackEditMode()).build();
            if (connection.suspended()) {
                connection.awake();
            }
            readline.readline(connection, prompt, line -> {
                out[0] = line.trim();
                connection.stopReading();
            }, completions);
            connection.openBlocking();
        } catch (IOException e) {
            System.out.println("Failed to read input from TerminalConnection");
            e.printStackTrace();
        }

        return out[0];
    }

    private static void initializeConnection() throws IOException {
        if (connection == null) {
            connection = new TerminalConnection();
            Consumer<Signal> sigintHandler = signal -> {
                if (signal == Signal.INT) {
                    connection.close();
                }
            };
            defaultCloseHandler = connection.getCloseHandler();
            Consumer<Void> closeHandler = signal -> {
                connection.write("Console installation aborted."); // localization
                connection.write(Config.getLineSeparator());
                InstallData.getInstance().setVariable("install.aborted", "true");
                Housekeeper housekeeper = Housekeeper.getInstance();
                housekeeper.runCleanup();
                housekeeper.wipeFiles();
                housekeeper.shutDownNoCleanUp(0);
            };
            Attributes attr = connection.getAttributes();
            attr.setLocalFlag(Attributes.LocalFlag.ECHOCTL, false);
            connection.setAttributes(attr);
            connection.setSignalHandler(sigintHandler);
            connection.setCloseHandler(closeHandler);
        }
    }

    /**
     * Used when the connection should be terminated in error.
     */
    public static void closeConnection(){
        if (connection != null){
            connection.close();
        }
    }

    /**
     * Resets the closeHandler to the one used at creation before stopping the console. Used when the installation is complete.
     */
    public static void closeConnectionClean() {
        if (connection != null) {
            connection.setCloseHandler(defaultCloseHandler);
            closeConnection();
        }
    }

    private static EditMode createIzpackEditMode() {
        EditMode izpackMode = EditModeBuilder.builder().create();
        // rebind defaults to use our custom Action instead of the default Enter() action
        IzPackEnter enterAction = new IzPackEnter();
        izpackMode.addAction(Key.CTRL_J, enterAction);
        izpackMode.addAction(Key.CTRL_M, enterAction);
        izpackMode.addAction(Key.ENTER, enterAction);
        izpackMode.addAction(Key.ENTER_2, enterAction);
        return izpackMode;
    }

    /**
     * Implement our own Enter action to prevent the prompt from moving to the next line:
     *  \ at the end of line
     *  open quotes
     *  lines starting with '#' being ignored
     */
    static class IzPackEnter implements Action {

        @Override
        public String name() {
            return "accept-line";
        }

        @Override
        public void accept(InputProcessor inputProcessor) {
            ConsoleBuffer console = inputProcessor.buffer();
            console.undoManager().clear();
            console.moveCursor(console.buffer().length());
            inputProcessor.setReturnValue(console.buffer().multiLine());
            console.buffer().reset();
        }
    }

    static class FilenameCompletion implements Completion<CompleteOperation> {

        private boolean isOnlyDirectories;

        FilenameCompletion(boolean onlyDirectories){
            this.isOnlyDirectories = onlyDirectories;
        }

        private String translatePath(String raw){
            String translatedInput;
            if (raw.startsWith("~" + File.separator)) {
                translatedInput = System.getProperty("user.home") + raw.substring(1);
            } else if (raw.startsWith("~")){
                String userName = raw.substring(1);
                translatedInput = Paths.get(Paths.get(System.getProperty("user.home")).getParent().toString(), userName).toString();
                translatedInput = userName.isEmpty() || raw.endsWith(File.separator) ? translatedInput + File.separator : translatedInput;
            } else if (!Paths.get(raw).isAbsolute()){
                translatedInput = Paths.get(System.getProperty("user.dir"), raw).toString();
            } else {
                translatedInput = raw;
            }
            return translatedInput;
        }

        @Override
        public void complete(CompleteOperation completeOperation) {
            completeOperation.doAppendSeparator(false);
            String rawInput = completeOperation.getBuffer();
            String translatedPath = translatePath(rawInput);

            Path currentPath = Paths.get(translatedPath);
            String incompleteFilename = getIncompleteFilename(translatedPath);
            List<String> filesInDirectory = new ArrayList<>();

            if (!translatedPath.isEmpty() && (Files.isDirectory(currentPath) || currentPath.getParent() != null)) {
                if (currentPath.getParent() == null && Files.isDirectory(currentPath)) {
                    if (OSUtils.IS_WINDOWS) {
                        Pattern drive = Pattern.compile("^[A-Z]:$");
                        Matcher driveMatcher = drive.matcher(currentPath.toString());
                        if (driveMatcher.matches()) {
                            completeOperation.addCompletionCandidate("\\");
                        } else {
                            filesInDirectory = getPossibleCompletions(currentPath, incompleteFilename);
                        }
                    } else {
                        filesInDirectory = getPossibleCompletions(currentPath, incompleteFilename);
                    }
                } else if (!Files.exists(currentPath) && currentPath.getParent() != null) {
                    if (OSUtils.IS_WINDOWS && !currentPath.toString().contains(File.separator) || translatedPath.endsWith(File.separator)) {
                        filesInDirectory = new ArrayList<>();
                    } else {
                        filesInDirectory = getPossibleCompletions(currentPath.getParent(), incompleteFilename);
                    }
                } else if (Files.exists(currentPath)) {
                    if (!incompleteFilename.isEmpty()) {
                        filesInDirectory = getPossibleCompletions(currentPath.getParent(), incompleteFilename);
                    } else {
                        filesInDirectory = getPossibleCompletions(currentPath, incompleteFilename);
                    }
                }
                completeOperation.addCompletionCandidates(filesInDirectory);
            }
            completeOperation.setOffset(rawInput.length() - incompleteFilename.length());
        }

        private List<String> getPossibleCompletions(Path currentPath, String incompleteFilename) {
            List<String> possibleCompletions;
            try {
                possibleCompletions = Files.list(currentPath).filter(isOnlyDirectories ? isDirectory() : path -> true)
                        .filter(isHiddenFile(incompleteFilename).negate())
                        .filter(startsWithIncompleteFilename(incompleteFilename))
                        .map(Path::getFileName)
                        .map(Path::toString)
                        .map(filename -> Files.isDirectory(currentPath.resolve(filename)) ? filename + File.separator : filename)
                        .collect(Collectors.toList());
            } catch (IOException e) {
                possibleCompletions = Collections.emptyList();
            }
            return possibleCompletions;
        }

        private String getIncompleteFilename(String currentPath) {
            if (currentPath.endsWith(File.separator)) {
                return "";
            } else {
                Path temp = Paths.get(currentPath).getFileName();
                return temp != null ? temp.toString() : "";
            }

        }

        private Predicate<Path> isDirectory(){
            return path -> Files.isDirectory(path);
        }

        private Predicate<Path> isHiddenFile(String incompleteFilename){
            return path -> incompleteFilename.isEmpty() && path.getFileName().toString().startsWith(".");
        }

        private Predicate<Path> startsWithIncompleteFilename(String incompleteFilename){
            return path -> path.getFileName().toString().startsWith(incompleteFilename);
        }
    }
}
