#!/usr/bin/env bash

set -e

function usage {
    echo "Usage:"
    echo "   create-versions.sh -v version -u updatable_versions -o output_directory"
    echo ""
    echo "   -v, --version <version>            Current components version."
    echo "   -u, --updatableVersions <list>     List of updatable components versions sorted in ascending order."
    echo "   -o, --outputDirectory <dir>        Output directory to place the version properties file."
}

while [[ $# -gt 0 ]]
do
    arg="$1"

    case "$arg" in
        -v|--version)
            shift
            CURRENT_VERSION="$1"
            ;;
        -u|--updatableVersions)
            shift
            UPDATABLE_VERSIONS="$1"
            ;;
        -o|--outputDirectory)
            shift
            OUTPUT_DIR="$1"
            ;;
        *)
            echo "Unknown argument: $1"
            usage
            exit 1
            ;;
    esac
    shift
done

if [[ -z "$CURRENT_VERSION" || -z "$UPDATABLE_VERSIONS" || -z "$OUTPUT_DIR" ]];
then
    usage
    exit 0
fi

UPDATABLE_VERSIONS=($(echo "${UPDATABLE_VERSIONS}" | tr ',' '\n'))
UPDATABLE_VERSION_MIN=${UPDATABLE_VERSIONS[0]}
UPDATABLE_VERSION_MAX=${UPDATABLE_VERSIONS[-1]}

VERSION_NAME=$(echo "${CURRENT_VERSION}" | sed -E 's/([0-9]+\.[0-9]+\.[0-9]+)\..*/\1/g')

function extract_community_version {
    productVersion="$1"
    mvn dependency:copy -Dartifact="org.kie.rhba:rhpam:${productVersion}:pom" -DoutputDirectory="${OUTPUT_DIR}"
    rhpamPOM="${OUTPUT_DIR}/rhpam-${productVersion}.pom"
    communityVersion=$(grep -e "<version.org.kie>.*</version.org.kie>" "${rhpamPOM}" | sed -E 's/.*>(.*)<\/.*/\1/g')
    if [[ "${communityVersion}" == "" ]];
    then
        mvn dependency:copy -Dartifact="org.kie.rhba:rhba-parent:${productVersion}:pom" -DoutputDirectory="${OUTPUT_DIR}"
        rhbaPOM="${OUTPUT_DIR}/rhba-parent-${productVersion}.pom"
        communityVersion=$(grep -e "<version.org.kie>.*</version.org.kie>" "${rhbaPOM}" | sed -E 's/.*>(.*)<\/.*/\1/g')
    fi
    rm -f "${rhpamPOM}" "${rhbaPOM}"
}

function extract_community_versions {
    extract_community_version "${UPDATABLE_VERSION_MIN}"
    COMMUNITY_VERSION_MIN="${communityVersion}"

    extract_community_version "${CURRENT_VERSION}"
    COMMUNITY_VERSION_MAX="${communityVersion}"
}

function create_version_properties {
    versionProperties="${OUTPUT_DIR}/version.properties"
    rm -f "${versionProperties}"
    echo "version.community.min=${COMMUNITY_VERSION_MIN}" >> "${versionProperties}"
    echo "version.community.max=${COMMUNITY_VERSION_MAX}" >> "${versionProperties}"
}

extract_community_versions
create_version_properties
