package main

import (
	"context"
	"fmt"
	"os"
	"strings"
	"time"

	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/runtime/serializer/json"
	"k8s.io/client-go/kubernetes/scheme"
	_ "k8s.io/client-go/plugin/pkg/client/auth"

	"github.com/spf13/cobra"

	"github.com/skupperproject/skupper/api/types"
)

func NewCmdToken() *cobra.Command {
	cmd := &cobra.Command{
		Use:   "token create <output-token-file> [--name <name>] or token status ...",
		Short: "Manage skupper tokens",
	}
	return cmd
}

var tokenType string
var password string
var expiry time.Duration
var uses int
var tokenTemplate string

func NewCmdTokenCreate(skupperClient SkupperTokenClient, flag string) *cobra.Command {
	subflag := ""
	if flag == "client-identity" {
		subflag = "i"
	} else if flag == "" {
		flag = "name" // default
	} else {
		panic("flag argument must be \"client-identity\" or \"\"")
	}
	cmd := &cobra.Command{
		Use:    "create <output-token-file>",
		Short:  "Create a token.  The 'link create' command uses the token to establish a link from a remote Skupper site.",
		Args:   cobra.ExactArgs(1),
		PreRun: skupperClient.NewClient,
		RunE:   skupperClient.Create,
	}
	cmd.Flags().StringVarP(&clientIdentity, flag, subflag, types.DefaultVanName, "Provide a specific identity as which connecting skupper installation will be authenticated")
	skupperClient.CreateFlags(cmd)
	return cmd
}

var TokenStatusOpts types.TokenStatus

func NewCmdTokenStatus(skupperClient SkupperTokenClient, flag string) *cobra.Command {
	cmd := &cobra.Command{
		Use:    "status <output-token-file>",
		Short:  "Displays information about the token, including its metadata, expiration data, and claim details.",
		Args:   cobra.ExactArgs(1),
		PreRun: skupperClient.NewClient,
		RunE: func(cmd *cobra.Command, args []string) error {
			silenceCobra(cmd)

			ctx, cancel := context.WithTimeout(context.Background(), types.DefaultTimeoutDuration)
			defer cancel()

			currentSiteId, err := skupperClient.GetCurrentSiteId(ctx)
			if err != nil && strings.HasPrefix(err.Error(), "Skupper is not enabled") {
				return fmt.Errorf("%s", err.Error())
			}
			// loading secret from file
			yaml, err := os.ReadFile(args[0])
			if err != nil || yaml == nil {
				return fmt.Errorf("could not read connection token %w", err)
			}

			ys := json.NewYAMLSerializer(json.DefaultMetaFactory, scheme.Scheme,
				scheme.Scheme)
			var secret = &corev1.Secret{}
			_, _, err = ys.Decode(yaml, nil, secret)
			if err != nil {
				return fmt.Errorf("could not parse connection token: %w", err)
			}
			TokenStatusOpts.Name = secret.ObjectMeta.Name
			TokenStatusOpts.Secret = secret
			if secret.ObjectMeta.Labels["skupper.io/type"] == types.TypeClaimRequest {
				TokenStatusOpts.Type = types.TokenRoleClaim
			} else {
				TokenStatusOpts.Type = types.TokenRoleCert
			}
			TokenStatusOpts.Creator = types.TokenCreatorRemote
			if currentSiteId == secret.ObjectMeta.Annotations[types.TokenGeneratedBy] {
				TokenStatusOpts.Creator = types.TokenCreatorLocal
			}
			fmt.Printf("Token was generated by %s site with id: %s\n", TokenStatusOpts.Creator, secret.ObjectMeta.Annotations[types.TokenGeneratedBy])
			fmt.Printf("\nToken Name: %s\n", TokenStatusOpts.Name)
			fmt.Printf("  path: %s\n", args[0])
			fmt.Printf("  type: %s\n", TokenStatusOpts.Type)
			fmt.Printf("  version: %s\n", secret.ObjectMeta.Annotations[types.SiteVersion])

			if TokenStatusOpts.Creator == types.TokenCreatorRemote {
				if TokenStatusOpts.Type == types.TokenRoleClaim {
					if secret.ObjectMeta.Annotations[types.ClaimExpiration] != "" {
						fmt.Printf("  expiry: %s\n", secret.ObjectMeta.Annotations[types.ClaimExpiration])
					}
				}
				return nil
			} else {
				return skupperClient.Status(cmd, args)
			}
		},
	}
	return cmd
}
