/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.couriers;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Properties;

import javax.jms.DeliveryMode;
import javax.jms.JMSException;
import javax.jms.MessageConsumer;
import javax.jms.MessageProducer;
import javax.jms.ObjectMessage;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicPublisher;
import javax.jms.TopicSession;
import javax.naming.Context;
import javax.naming.NamingException;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.helpers.JmsComposer;
import org.jboss.internal.soa.esb.rosetta.pooling.ConnectionException;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPool;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPoolContainer;
import org.jboss.internal.soa.esb.rosetta.pooling.xa.XaJmsConnectionPool;
import org.jboss.internal.soa.esb.rosetta.pooling.xa.XaJmsConnectionPoolContainer;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierTimeoutException;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.helpers.NamingContextException;
import org.jboss.soa.esb.helpers.NamingContextPool;
import org.jboss.soa.esb.listeners.gateway.DefaultESBPropertiesSetter;
import org.jboss.soa.esb.listeners.gateway.ESBPropertiesSetter;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.notification.jms.DefaultJMSPropertiesSetter;
import org.jboss.soa.esb.notification.jms.JMSPropertiesSetter;
import org.jboss.soa.esb.util.Util;

public class JmsCourier implements PickUpOnlyCourier, DeliverOnlyCourier {
	
    /**
     * package protected constructor - Objects of Courier should only be
     * instantiated by the Factory
     *
     * @param epr
     */
    public JmsCourier(JMSEpr epr) throws CourierException {
        this(epr, false);
    }

    /**
     * package protected constructor - Objects of Courier should only be
     * instantiated by the Factory
     *
     * @param epr
     */
    public JmsCourier(JMSEpr epr, boolean isReceiver) throws CourierException {
        _isReceiver = isReceiver;
        _epr = epr;
        _sleepForRetries = 3000;

        if (!_isReceiver) {
            try {
                _messageProperties = Util.propertiesFromSelector(_epr
                        .getMessageSelector());
            }
            catch (Exception e) {
                throw new CourierException(e);
            }
        }
    } // ________________________________

    public void cleanup() {
        synchronized(this) {
            if (_messageProducer != null) {
                try {
                    _messageProducer.close();
                } catch (Exception e) {
                    _logger.debug(e.getMessage(), e);
                } finally {
                    _messageProducer = null;
                    closeSession();
                }
            }

            if (_messageConsumer != null) {
                try {
                   _messageConsumer.close();
                } catch (JMSException e) {
                    _logger.debug(e.getMessage(), e);
                } finally {
                    _messageConsumer = null;
                    closeSession();
                }
            }
        }
    } // ________________________________

    private void closeSession() {
        synchronized(this) {
            if (jmsSession != null) {
                try {
                    getConnectionPool().closeSession(jmsSession);
                } catch (ConnectionException e) {
                    _logger.error("Unable to get Connection Poll for closing of JMS Session.", e);
                } finally {
                    jmsSession = null;
                }
            }
        }
    }
    
    public Session getJmsSession() throws CourierException {
    	return getJmsSession(Session.AUTO_ACKNOWLEDGE);
    }

    public Session getJmsSession(final int acknowledgeMode) throws CourierException {
	reset();
	
        if(jmsSession == null) {
            synchronized(this) {
        	checkTransaction();
        	
                if(jmsSession == null) {
                    String sType;

                    try {
                        sType = _epr.getDestinationType();
                    } catch (URISyntaxException e) {
                        throw new CourierException("EPR.getDestinationType failed.", e);
                    }

                    try {
                        if (JMSEpr.QUEUE_TYPE.equals(sType)) {
                            jmsSession = getConnectionPool().getQueueSession(acknowledgeMode);
                        } else {
                            jmsSession = getConnectionPool().getTopicSession(acknowledgeMode);
                        }
                    } catch (NamingException e) {
                        throw new CourierException("Failed to get JMS Session from pool.", e);
                    } catch (JMSException e) {
                        throw new CourierException("Failed to get JMS Session from pool.", e);
                    } catch (ConnectionException e) {
                        throw new CourierException("Failed to get JMS Session from pool.", e);
                    }
                }
            }
        }

        return jmsSession;
    }

    /**
     * package the ESB message in a javax.jms.ObjectMessage, and send it
     *
     * @param message Message - the message to deliverAsync
     * @return boolean - the result of the delivery
     * @throws CourierException -
     *                          if problems were encountered
     */
    public boolean deliver(org.jboss.soa.esb.message.Message message) throws CourierException {
        ObjectMessage msg;

        if (null == message) {
            return false;
        }
        
        if (_messageProducer == null) {
            try {
                createMessageProducer();
            } catch (final NamingContextException nce) {
                throw new CourierException("Unexpected exception attempting to access naming context pool", nce) ;
            }
        }

        // Create the JMS message from the serialized ESB message...
        try {
            msg = getJmsSession(_epr.getAcknowledgeMode()).createObjectMessage(Util.serialize(message));
        } catch (JMSException e) {
            throw new CourierException("Failed to serialize ESB Message.", e);
        } catch (ParserConfigurationException e) {
            throw new CourierException("Failed to serialize ESB Message.", e);
        } catch (IOException e) {
            throw new CourierException("Failed to serialize ESB Message.", e);
        }

        // Set the JMS message from the ESB message...
        try {
            setJMSProperties(message, msg);
        } catch (JMSException e) {
            throw new CourierException("Failed to set JMS Message properties from ESB Message properties.", e);
        }

        return deliver(msg);
    }

    /**
     * Send the JMS message.
     *
     * @param message Message - the message to deliverAsync
     * @return boolean - the result of the delivery
     * @throws CourierException -
     *                          if problems were encountered
     */
    public boolean deliver(javax.jms.Message message) throws CourierException {
        if (_isReceiver) {
            throw new CourierException("This is a read-only Courier");
        }

        if (null == message) {
            return false;
        }
	
	reset();
	
        if (_messageProducer == null) {
            try {
                createMessageProducer();
            } catch (final NamingContextException nce) {
                throw new CourierException("Unexpected exception attempting to access naming context pool", nce) ;
            }
        }

        while (null != _messageProducer) {
            try {
                for (KeyValuePair kvp : _messageProperties) {
                    String key = kvp.getKey();
                    if(message.getStringProperty(key) == null) {
                        message.setStringProperty(key, kvp.getValue());
                    }
                }

                checkTransaction();
                
                sendMessage(message);
                if ( jmsSession.getTransacted() && !transactional )
	                jmsSession.commit();
                
                return true;
            }
            catch (JMSException e) {
                jmsConnectRetry(e);
            }
            catch (Exception e) {
                throw new CourierException(e);
            }
        }
        return false;
    } // ________________________________


    /**
     * send/publish a javax.jms.ObjectMessage (that will contain the serialized
     * ESB Message)
     *
     * @param jmsMessage
     */
    private void sendMessage(javax.jms.Message jmsMessage) throws JMSException, URISyntaxException {
        String sType = _epr.getDestinationType();
        if (JMSEpr.TOPIC_TYPE.equals(sType)) {
            ((TopicPublisher) _messageProducer).publish(jmsMessage);
        } else {
            _messageProducer.send(jmsMessage);
        }

    } // ________________________________

    private void jmsConnectRetry(Exception exc) {
        _logger.debug("JMS error.  Attempting JMS reconnect.", exc);

        synchronized(this) {
            cleanup();

            final int maxRetry = 5; // TODO Magic number here!!!
            for (int i1 = 0; i1 < maxRetry; i1++) {
                // try to reconnect to the queue
                try {
                    if (_isReceiver) createMessageConsumer();
                    else
                        createMessageProducer();
                    break;
                }
                catch (Exception e) {
                    if (i1 < maxRetry - 1) {
                        try {
                            Thread.sleep(_sleepForRetries);
                        }
                        catch (InterruptedException e1) { // Just return after logging
                            _logger.debug("Unexpected thread interupt exception.", e);
                            break;
                        }
                    } else {
                        _logger.debug("Failed to reconnect to JMS", e);
                    }
                }
            }
        }
    } // ________________________________

    private void createMessageProducer() throws CourierException, NamingContextException {
        Context oJndiCtx = null;

        reset();
	
        checkTransaction();
        
        if (_messageProducer == null) {
            synchronized(this) {
                if (_messageProducer == null) {
                    try {
                        oJndiCtx = NamingContextPool.getNamingContext(_epr.getJndiEnvironment());

                        String sType = _epr.getDestinationType();
                        if (JMSEpr.QUEUE_TYPE.equals(sType)) {
                            QueueSession qSess = (QueueSession) getJmsSession(_epr.getAcknowledgeMode());
                            javax.jms.Queue queue = null;
                            try {
                                queue = (javax.jms.Queue) oJndiCtx.lookup(_epr
                                        .getDestinationName());
                            } catch (NamingException ne) {
                                try {
                                    oJndiCtx = NamingContextPool.replaceNamingContext(oJndiCtx, _epr.getJndiEnvironment());
                                    queue = (javax.jms.Queue) oJndiCtx.lookup(_epr
                                            .getDestinationName());
                                } catch (NamingException nex) {
                                    //ActiveMQ
                                    queue = qSess.createQueue(_epr.getDestinationName());
                                }
                            }
                            _messageProducer = qSess.createSender(queue);
                        } else if (JMSEpr.TOPIC_TYPE.equals(sType)) {
                            TopicSession tSess = (TopicSession) getJmsSession(_epr.getAcknowledgeMode());
                            Topic topic = null;
                            try {
                                topic = (Topic) oJndiCtx.lookup(_epr
                                        .getDestinationName());
                            }
                            catch (NamingException ne) {
                                topic = tSess.createTopic(_epr.getDestinationName());
                            }
                            _messageProducer = tSess.createPublisher(topic);
                        } else {
                            throw new CourierException("Unknown destination type");
                        }
                        _messageProducer.setDeliveryMode(_epr.getPersistent()?DeliveryMode.PERSISTENT:DeliveryMode.NON_PERSISTENT);
                        if ( _logger.isDebugEnabled() )
                            _logger.debug("JMSCourier deliveryMode: " + _messageProducer.getDeliveryMode() + ", peristent:" + _epr.getPersistent());
                    }
                    catch (JMSException ex) {
                        _logger.debug("Error from JMS system.", ex);

                        throw new CourierException(ex);
                    }
                    catch (URISyntaxException ex) {
                        throw new CourierException(ex);
                    } finally {
                        if (oJndiCtx != null) {
                            NamingContextPool.releaseNamingContext(oJndiCtx) ;
                        }
                    }
                }
            }
        }
    } // ________________________________

    private JmsConnectionPool getConnectionPool() throws ConnectionException {
	reset();
	
        if (jmsConnectionPool == null) {
            synchronized(this) {
        	try
        	{
        	    checkTransaction();
        	}
        	catch (CourierException ex)
        	{
        	    throw new ConnectionException(ex);
        	}
        	
                if(jmsConnectionPool == null) {
                    String sFactoryClass;
                    String sType;
                    Properties properties;
                    String username;
                    String password;
                    boolean transacted;

                    try {
                        sFactoryClass = _epr.getConnectionFactory();
                        sType = _epr.getDestinationType();
                        properties = _epr.getJndiEnvironment();
                        username = _epr.getJMSSecurityPrincipal();
                        password = _epr.getJMSSecurityCredential();
                        transacted = _epr.getTransacted();
                    } catch (URISyntaxException e) {
                        throw new ConnectionException("Unexpected exception while getting JMS connection pool.", e);
                    }

                    if (Util.isNullString(sFactoryClass)) {
                        sFactoryClass = "ConnectionFactory";
                    }

                    /*
                     * Needs to be a one-shot instance if transactional.
                     */
                    
                    Object tx = null;
                    
                    try
                    {
                	TransactionStrategy txS = TransactionStrategy.getTransactionStrategy(true);
                	
                	tx = txS.getTransaction();
                    }
                    catch (TransactionStrategyException ex)
                    {
                	_logger.warn("Problem getting transaction strategy: ", ex);
                	
                	throw new ConnectionException(ex);
                    }

                    if (tx == null)
                	jmsConnectionPool = JmsConnectionPoolContainer.getPool(properties, sFactoryClass, sType, username, password, transacted);
                    else
                	jmsConnectionPool = XaJmsConnectionPoolContainer.getPool(properties, sFactoryClass, sType, username, password, true, tx);  // force transacted to be true!
                }
            }
        }

        return jmsConnectionPool;
    }

    private void reset ()
    {
	/*
	 * Are we in a global transaction?
	 */
	
	if (jmsConnectionPool instanceof XaJmsConnectionPool)
	{
	    /*
	     * If the global transaction has terminated then this pool instance has been
	     * closed and we need to get another one.
	     */
	    
	    if (!((XaJmsConnectionPool) jmsConnectionPool).active())
	    {
		jmsConnectionPool = null;
		_messageProducer = null;
		_messageConsumer = null;
		jmsSession = null;
	    }
	}
    }
    
    private void checkTransaction () throws CourierException
    {
	try
	{
        	TransactionStrategy txStrategy = TransactionStrategy.getTransactionStrategy(true);
		Object txHandle = txStrategy.getTransaction();
		boolean isActive = txStrategy.isActive();
		
		transactional = (txHandle != null);
		
		/*
		 * Make sure the current transaction is still active! If we
		 * have previously slept, then the timeout may be longer than that
		 * associated with the transaction.
		 */
		
		if (transactional && !isActive)
		{
			throw new CourierException("Associated transaction is no longer active!");
		}
	}
	catch (TransactionStrategyException ex)
	{
	    throw new CourierException(ex);
	}
    }
    
    public Message pickup(long millis) throws CourierException, CourierTimeoutException {
        javax.jms.Message jmsMessage = pickupPayload(millis);

        return JmsComposer.compose(jmsMessage, esbPropertiesStrategy) ;
    }

    public javax.jms.Message pickupPayload(long millis) throws CourierException, CourierTimeoutException {
        if (!_isReceiver)
            throw new CourierException("This is an outgoing-only Courier");
        if (millis < 1)
            throw new IllegalArgumentException("Timeout millis must be > 0");
        if (null == _messageConsumer) try {
            createMessageConsumer();
        }
        catch (Exception e) {
            try {
                Thread.sleep(1000); // TODO magic number
            }
            catch (InterruptedException eI) {/* OK do nothing */
            }
            throw new CourierException("Unable to create Message Consumer", e);
        }

        javax.jms.Message jmsMessage = null;
        while (null != _messageConsumer) {
            checkTransaction();
            
            try {
                jmsMessage = _messageConsumer.receive(millis);
                break;
            }
            catch (JMSException e) {
                jmsConnectRetry(e);
            }
            catch (Exception e) {
                throw new CourierException(e);
            }
        }
        return jmsMessage;
    } // ________________________________

    /**
     * Sets the strategy for handling the setting of properties on an outgoing
     * JMS Message.
     *
     * @param jmsPropertiesStrategy the strategy to use.
     */
    public void setJmsPropertiesStrategy(JMSPropertiesSetter jmsPropertiesStrategy) {
        this.jmsPropertiesStrategy = jmsPropertiesStrategy;
    }

    /**
     * Set the {@link ESBPropertiesSetter} to be used
     *
     * @param esbPropertiesStrategy the strategy to be used
     */
    public void setEsbPropertiesStrategy(ESBPropertiesSetter esbPropertiesStrategy) {
        this.esbPropertiesStrategy = esbPropertiesStrategy;
    }

    /**
     * This method will set appropriate JMSProperties on the outgoing JMSMessage object.
     * </p>
     * Sublclasses can either override this method add a different behaviour, or they can
     * set the strategy by calling {@link #setJmsPropertiesStrategy(JMSPropertiesSetter)}.
     * </p>
     * See {@link org.jboss.soa.esb.notification.jms.JMSPropertiesSetter} for more info.
     */
    protected void setJMSProperties(org.jboss.soa.esb.message.Message fromESBMessage, javax.jms.Message toJMSMessage) throws JMSException {
        jmsPropertiesStrategy.setJMSProperties(fromESBMessage, toJMSMessage);
    }

    /**
     * Delegates to {@link DefaultESBPropertiesSetter#setPropertiesFromJMSMessage(javax.jms.Message,Message)}
     * by default, but this method can be overridden by subclasses that need a different behaviour.
     * </p>
     * It is also possible to set a different strategy by setting {@link #setEsbPropertiesStrategy(ESBPropertiesSetter)}
     */
    protected void setPropertiesFromJMSMessage(javax.jms.Message fromJMS, org.jboss.soa.esb.message.Message toESB) throws JMSException {
        esbPropertiesStrategy.setPropertiesFromJMSMessage(fromJMS, toESB);
    }

    private void createMessageConsumer() throws CourierException, ConfigurationException, MalformedEPRException, NamingContextException {
        Context oJndiCtx = null;

        reset();
        
        if (_messageConsumer == null) {
            synchronized(this) {
                if (_messageConsumer == null) {
                    boolean success = false;
                    try {
                        Properties environment = _epr.getJndiEnvironment();
                        oJndiCtx = NamingContextPool.getNamingContext(environment);
                        try
                        {
                            String sType = _epr.getDestinationType();
                            if (JMSEpr.QUEUE_TYPE.equals(sType)) {
                                QueueSession qSess = (QueueSession) getJmsSession(_epr.getAcknowledgeMode());
                                javax.jms.Queue queue = null;
                                try {
                                    queue = (javax.jms.Queue) oJndiCtx.lookup(_epr
                                            .getDestinationName());
                                } catch (NamingException ne) {
                                    try {
                                        oJndiCtx = NamingContextPool.replaceNamingContext(oJndiCtx, environment);
                                        queue = (javax.jms.Queue) oJndiCtx.lookup(_epr
                                                .getDestinationName());
                                    } catch (NamingException nex) {
                                        //ActiveMQ
                                        queue = qSess.createQueue(_epr.getDestinationName());
                                    }
                                }
                                _messageConsumer = qSess.createReceiver(queue, _epr.getMessageSelector());
                            } else if (JMSEpr.TOPIC_TYPE.equals(sType)) {
                                TopicSession tSess = (TopicSession) getJmsSession(_epr.getAcknowledgeMode());
                                Topic topic = tSess.createTopic(_epr.getDestinationName());
                                _messageConsumer = tSess.createConsumer(topic, _epr
                                        .getMessageSelector());
                            } else {
                                throw new CourierException("Unknown destination type");
                            }
                            success = true;
                        } finally {
                            NamingContextPool.releaseNamingContext(oJndiCtx) ;
                        }
                    }
                    catch (JMSException ex) {
                        _logger.debug("Error from JMS system.", ex);

                        throw new CourierException(ex);
                    }
                    catch (URISyntaxException ex) {
                        throw new MalformedEPRException(ex);
                    }
                    finally {
                        if (!success) {
                            closeSession();
                        }
                    }
                }
            }
        }
    } // ________________________________

    long _sleepForRetries = 3000; // milliseconds

    protected boolean _isReceiver;

    protected JMSEpr _epr;

    protected Logger _logger = Logger.getLogger(JmsCourier.class);

    protected String _messageSelector;

    protected volatile Session jmsSession;

    protected volatile MessageProducer _messageProducer;

    protected volatile MessageConsumer _messageConsumer;

    protected List<KeyValuePair> _messageProperties;

    protected volatile JmsConnectionPool jmsConnectionPool;

    private boolean transactional = false;
    
    /**
     * Strategy for setting JMSProperties
     */
    private JMSPropertiesSetter jmsPropertiesStrategy = new DefaultJMSPropertiesSetter();

    /**
     * Strategy for setting JMS Properties on the ESB Message object created
     * by the process method.
	 */
	private ESBPropertiesSetter esbPropertiesStrategy = new DefaultESBPropertiesSetter();
	
}
