/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.couriers;

import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.addressing.eprs.JDBCEpr;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierTimeoutException;
import org.jboss.soa.esb.helpers.persist.JdbcCleanConn;
import org.jboss.soa.esb.helpers.persist.SimpleDataSource;
import org.jboss.soa.esb.listeners.message.errors.Factory;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.util.Type;
import org.jboss.soa.esb.util.Util;
import org.xml.sax.SAXParseException;

public class SqlTableCourier implements PickUpOnlyCourier, DeliverOnlyCourier
{
	/**
	 * disable default constructor
	 */
	private SqlTableCourier()
	{
	}

	/**
	 * package protected constructor - Objects of Courier should only be
	 * instantiated by the Factory
	 * 
	 * @param epr
	 */
	SqlTableCourier(JDBCEpr epr) throws CourierException
	{
		this(epr, false);
	}

	/**
	 * package protected constructor - Objects of Courier should only be
	 * instantiated by the Factory
	 * 
	 * @param epr
	 */
	SqlTableCourier(JDBCEpr epr, boolean isReceiver) throws CourierException
	{
		_isReceiver = isReceiver;
		_epr = epr;
		_sleepForRetries = 3000;  // TODO magic number - configurable?
		try
		{
			_postDelete = Boolean.TRUE.equals(Boolean.valueOf(epr
					.getPostDelete()));
			_errorDelete = Boolean.TRUE.equals(Boolean.valueOf(epr
					.getErrorDelete()));
		}
		catch (URISyntaxException e)
		{
			throw new CourierException(e);
		}

	} // ________________________________

	public void cleanup()
	{
		if (null != _conn)
		{
			try
			{
				_conn.release();
			}
			catch (Exception e)
			{
				e.printStackTrace();
				_logger.info("Unable to release connection", e);
			}
		}

	} // ________________________________

	/**
	 * package the ESB message in a java.io.Serializable, and write it.
	 * Delivery occurs within its own transaction if there is no
	 * global transaction active.
	 * 
	 * @param message
	 *            Message - the message to deliverAsync
	 * @return boolean - the result of the delivery
	 * @throws CourierException -
	 *             if problems were encountered
	 */
	
	public boolean deliver(Message message) throws CourierException
	{
		if (_isReceiver)
			throw new CourierException("This is a read-only Courier");

		if (null == message)
			return false;

		String msgId = null;
		Call call = message.getHeader().getCall();
		if (null==call)
			message.getHeader().setCall(call=new Call());
		try
		{
			if (null==call.getMessageID())
				call.setMessageID(new URI(UUID.randomUUID().toString()));
			msgId = call.getMessageID().toString();
		}
		catch (URISyntaxException e)
		{
			throw new CourierException("Problems with message header ",e);
		}

		try
		{
        		TransactionStrategy txStrategy = TransactionStrategy.getStrategy();
        		Object txHandle = ((txStrategy == null) ? null : txStrategy.getTransaction());
        		boolean isActive = ((txStrategy == null) ? false : txStrategy.isActive());
        		
        		transactional = (txHandle != null);
        		
        		/*
        		 * Make sure the current transaction is still active! If we
        		 * have previously slept, then the timeout may be longer than that
        		 * associated with the transaction.
        		 */
        		
        		if (transactional && !isActive)
        		{
        			throw new CourierException("Associated transaction is no longer active!");
        		}
		}
		catch (TransactionStrategyException ex)
		{
		    throw new CourierException(ex);
		}
		
		if (null == _conn)
		{
			try
			{
				_conn = getConn();
			}
			catch (Exception e)
			{
				throw new CourierException(e);
			}
		}

		while (_conn != null)
		{
			try
			{
				int iCol = 1;
				PreparedStatement PS = insertStatement();
				PS.setString(iCol++, msgId);
				PS.setObject(iCol++, Util.serialize(message));
				PS.setString(iCol++, State.Pending.getColumnValue());
				PS.setLong(iCol++, System.currentTimeMillis());

				_conn.execUpdWait(PS, 3);
				
				if (!transactional)
				    _conn.commit();
				
				return true;
			}
			catch (SQLException e)
			{
				if (null != _conn)
				{
					try
					{
					    if (!transactional)
						_conn.rollback();
					}
					catch (Exception roll)
					{
						_logger.debug(roll);
					}
				}
				
				_logger.debug("SQL exception during deliver", e);
				throw new CourierException(e);
			}
			catch (Exception e)
			{
				jdbcConnectRetry(e);
			}
		}
		return false;
	} // ________________________________

	public Message pickup(long millis) throws CourierException, CourierTimeoutException
	{
		Message result = null;
		long limit = System.currentTimeMillis()
				+ ((millis < 100) ? 100 : millis);
		
		do
		{
			try
			{
				TransactionStrategy txStrategy = TransactionStrategy.getStrategy();
				Object txHandle = ((txStrategy == null) ? null : txStrategy.getTransaction());
				boolean isActive = ((txStrategy == null) ? false : txStrategy.isActive());
				
				transactional = (txHandle != null);
				
				/*
				 * Make sure the current transaction is still active! If we
				 * have previously slept, then the timeout may be longer than that
				 * associated with the transaction.
				 */
				
				/*
				 * MessageAwareListener will catch exceptions and roll back the transaction.
				 */
				
				if (transactional && !isActive)
				{
					throw new CourierException("Associated transaction is no longer active!");
				}
			}
			catch (TransactionStrategyException ex)
			{
				_logger.error("Could not determine transaction association!", ex);
				
				throw new CourierException("Could not determine transaction association!");
			}
			
			ResultSet RS = null;
			
			try
			{
			    RS = getRowList();

			    while (null != RS && RS.next())
			    {
				String messageId = RS.getString(1);

				if (null == (result = tryToPickup(messageId)))
				    continue;

				/*
				 * If this is fault message, then throw an exception with the contents. With the
				 * exception of user-defined exceptions, faults will have nothing in the body, properties etc.
				 */

				if (Type.isFaultMessage(result))
				    Factory.createExceptionFromFault(result);

				return result;
			    }
			}
			catch (SQLException e)
			{
			    _logger.debug("SQL Exception during pickup", e);
			    return null;
			}
			finally
			{
			    try
			    {
        			    if (RS != null)
        				RS.close();
			    }
			    catch (final SQLException ex)
			    {
				_logger.warn("SQLException during close of ResultSet.", ex);
			    }
			    
			    // Added to make sure we release transactions from all paths
			    if (_conn != null)
			    {
				try
				{
				    if (!transactional)
					_conn.rollback() ;
				}
				catch (final SQLException sqle) {} //ignore
			    }
			}
			
			try
			{
			    long lSleep = limit - System.currentTimeMillis();
			    if (_pollLatency < lSleep)
				lSleep = _pollLatency;
			    if (lSleep > 0)
				Thread.sleep(lSleep);
			}
			catch (InterruptedException e)
			{
			    return null;
			}
		} while (System.currentTimeMillis() <= limit);
		return null;
	} // ________________________________

	private Message tryToPickup(String messageId) throws CourierException,
			SQLException
	{
		int iParm = 1;

		select4UpdateStatement().setString(iParm++, messageId);
		select4UpdateStatement().setString(iParm++,
				State.Pending.getColumnValue());

		while (_conn != null)
		{
		    ResultSet RS = null;
		    
			try
			{
				RS = _conn.execQueryWait(select4UpdateStatement(), 3);
				
				while (RS.next())
				{
					Exception eBad = null;
					try
					{
						Message result = Util.deserialize((Serializable) RS
								.getObject(1));
						if (_postDelete)
							deleteMsg(messageId);
						else
							changeStatus(messageId, State.Done);
						return result;
					}
					catch (ClassCastException e)
					{
						eBad = e;
					}
					catch (SAXParseException e)
					{
						eBad = e;
					}
					catch (Exception e)
					{
						throw new CourierException(e);
					}
					if (null != eBad)
					{
						if (_errorDelete)
							deleteMsg(messageId);
						else
							changeStatus(messageId, State.Error);
						continue;
					}
				}
				return null;
			}
			catch (SQLException e)
			{
				throw new CourierException(e);
			}
			catch (Exception ex)
			{
				jdbcConnectRetry(ex);
			}
			finally
			{
			    try
			    {
				if (RS != null)
				    RS.close();
			    }
			    catch (final Exception ex)
			    {
				_logger.warn("Could not close ResultSet.", ex);
			    }
			}
		}
		return null;
	} // ________________________________

	private void deleteMsg(String messageId) throws SQLException
	{
		int iParm = 1;
		deleteStatement().setString(iParm++, messageId);
		_conn.execUpdWait(deleteStatement(), 3);
		
		if (!transactional)
			_conn.commit();
	}

	private void changeStatus(String messageId, State to) throws SQLException
	{
		int iParm = 1;
		updateStatusStatement().setString(iParm++, to.getColumnValue());
		updateStatusStatement().setString(iParm++, messageId);
		_conn.execUpdWait(updateStatusStatement(), 3);
		
		if (!transactional)
			_conn.commit();
	}

	private ResultSet getRowList() throws CourierException
	{
		if (null == _conn)
		{
			try
			{
				_conn = getConn();
			}
			catch (Exception e)
			{
				throw new CourierException(e);
			}
		}
		while (_conn != null)
		{
			try
			{
				return _conn.execQueryWait(listStatement(), 3);
			}
			catch (Exception e)
			{
				_logger.debug("Problem encountered while executing query.", e);
				e.printStackTrace();
				
				jdbcConnectRetry(e);
			}
		}
		return null;

	} // _______________________________

	private void jdbcConnectRetry(Exception exc)
	{
		_logger.debug("DB problem, will try to reconnect", exc);
		
		cleanup();
		_conn = null;

		_prepDelete = _prepGetList = _prepInsert = _prepSel4Upd = _prepUpdateStatus = null;
		for (int i1 = 0; i1 < 3; i1++)
		{
			try
			{
				_conn = getConn();
			}
			catch (Exception e)
			{
				try
				{
					Thread.sleep(_sleepForRetries);
				}
				catch (InterruptedException eInt)
				{
					return;
				}
			}
		}
	} // ________________________________

	private JdbcCleanConn getConn() throws SQLException, MalformedEPRException, NamingException
	{
		if (null == _conn)
		{
			try
			{
				DataSource DS = null;
				if (_epr.getDatasource() == null) {
					DS = new SimpleDataSource(_epr.getDriver(), 
						_epr.getURL(), _epr.getUserName(), _epr.getPassword());
				} else {
					InitialContext initContext;
					try {
						initContext = new InitialContext();
						DS = (DataSource) initContext.lookup(_epr.getDatasource());
					} catch (NamingException e) {
						_logger.error("Problem resolving DataSource through JNDI", e);
						
						throw e; // it'll get wrapped later anyway!
					}
				}
				_conn = new JdbcCleanConn(DS, transactional);
			}
			catch (URISyntaxException ex)
			{
				throw new MalformedEPRException(ex);
			}
		}
		return _conn;
	} // ________________________________

	protected PreparedStatement listStatement() throws SQLException
	{
		if (null == _prepGetList)
		{
			try
			{
				String[] columns =
				{ _epr.getMessageIdColumn(), _epr.getTimestampColumn() };

				StringBuilder sb = new StringBuilder("select");
				int i1 = 0;
				for (String col : columns)
					sb.append((i1++ < 1) ? " " : ",").append(col);
				sb.append(" from ").append(_epr.getTableName());
				sb.append(" where ").append(_epr.getStatusColumn())
						.append("='").append(State.Pending.getColumnValue())
						.append("'").append(" order by 2");
				_prepGetList = getConn().prepareStatement(sb.toString());
			}
			catch (SQLException ex)
			{
				throw ex;
			}
			catch (Exception e)
			{
				e.printStackTrace();
				_logger.debug("Unable to prepare SQL statement", e);
				throw new SQLException("Problem encountered when trying to created PreparedStatement: "+e);
			}
		}
		
		return _prepGetList;
	} // ________________________________

	protected PreparedStatement select4UpdateStatement()
	{
		if (_prepSel4Upd == null)
		{
			try
			{
				/*
				 * TODO make this dynamic using a factory pattern.
				 */

				StringBuilder sb = null;

				if (!_epr.getURL().contains("hsqldb"))
				{
					sb = new StringBuilder("select ").append(
							_epr.getDataColumn()).append(" from ").append(
							_epr.getTableName()).append(" where ").append(
							_epr.getMessageIdColumn()).append("=?").append(
							" and ").append(_epr.getStatusColumn())
							.append("=?").append(" for update");
				}
				else
				{
					/*
					 * HSQL does not support FOR UPDATE! All tables appear to
					 * be inherently updatable!
					 */
					
					sb = new StringBuilder("select ").append(
							_epr.getDataColumn()).append(" from ").append(
							_epr.getTableName()).append(" where ").append(
							_epr.getMessageIdColumn()).append("=?").append(
							" and ").append(_epr.getStatusColumn())
							.append("=?");
				}

				_prepSel4Upd = getConn().prepareStatement(sb.toString());
			}
			catch (Exception e)
			{
				_logger.debug(e);
				return null;
			}
		}

		return _prepSel4Upd;
	} // ________________________________

	protected PreparedStatement updateStatusStatement()
	{
		if (null == _prepUpdateStatus)
			try
			{
				StringBuilder sb = new StringBuilder("update ").append(
						_epr.getTableName()).append(" set ").append(
						_epr.getStatusColumn()).append("= ?").append(" where ")
						.append(_epr.getMessageIdColumn()).append("=?");
				_prepUpdateStatus = getConn().prepareStatement(sb.toString());
			}
			catch (Exception e)
			{
				_logger.debug(e);
				return null;
			}
		return _prepUpdateStatus;
	} // ________________________________

	protected PreparedStatement insertStatement()
	{
		if (null == _prepInsert)
			try
			{
				String[] columns =
				{ _epr.getMessageIdColumn(), _epr.getDataColumn(),
						_epr.getStatusColumn(), _epr.getTimestampColumn() };

				StringBuilder sb = new StringBuilder("insert into ").append(
						_epr.getTableName()).append("(");
				int i1 = 0;
				for (String col : columns)
					sb.append((i1++ < 1) ? " " : ",").append(col);
				sb.append(") values (?,?,?,?)");
				_prepInsert = getConn().prepareStatement(sb.toString());
			}
			catch (Exception e)
			{
				_logger.debug(e);
				return null;
			}
		return _prepInsert;
	} // ________________________________

	protected PreparedStatement deleteStatement()
	{
		if (null == _prepDelete)
			try
			{
				StringBuilder sb = new StringBuilder("delete from ").append(
						_epr.getTableName()).append(" where ").append(
						_epr.getMessageIdColumn()).append(" =?");
				_prepDelete = getConn().prepareStatement(sb.toString());
			}
			catch (Exception e)
			{
				_logger.debug(e);
				return null;
			}
		return _prepDelete;
	} // ________________________________

	protected enum State
	{
		Pending, WorkInProgress, Done, Error;
		String getColumnValue()
		{
			return toString().substring(0, 1);
		}
	}

	public void setPollLatency(Long millis)
	{
		if (millis <= 200)
			_logger.warn("Poll latency must be >= 200 milliseconds - Keeping old value of "+_pollLatency);
		else
			_pollLatency = millis;
	} // ________________________________
	
	protected long _pollLatency = 200;
	protected long _sleepForRetries = 3000; // milliseconds

	protected boolean _postDelete, _errorDelete;
	protected boolean _isReceiver;

	protected JDBCEpr _epr;

	protected JdbcCleanConn _conn;

	protected PreparedStatement _prepGetList;
	protected PreparedStatement _prepSel4Upd;
	protected PreparedStatement _prepUpdateStatus;
	protected PreparedStatement _prepInsert;
	protected PreparedStatement _prepDelete;
	
	private boolean transactional = false;

	protected static Logger _logger = Logger.getLogger(SqlTableCourier.class);
}