/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.message.filter;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.util.Calendar;
import java.util.Map;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.FTPEpr;
import org.jboss.soa.esb.addressing.eprs.FileEpr;
import org.jboss.soa.esb.addressing.eprs.HibernateEpr;
import org.jboss.soa.esb.addressing.eprs.JDBCEpr;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.filter.InputOutputFilter;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.Properties;

/**
 * An inputoutput filter is called when the message is being received/sent and
 * after it has been received from the transport/before it is passed to the
 * transport. Implementations may augment or modify the message en route.
 * Implementations that are pure output or pure input can just override the
 * desired operation.
 * 
 * This one adds some metadata to the message as it flows through the ESB.
 * 
 * @author marklittle
 */

public class MetaDataFilter extends InputOutputFilter
{
    public Message onOutput (Message msg, Map<String, Object> params) throws CourierException
    {
	final Environment.Transports type;
	final String name;
	EPR destination = msg.getHeader().getCall().getTo();
	if (destination != null)
	{
	    if (destination instanceof FTPEpr)
	    {
		try
		{
		    type = Environment.Transports.FTP;

		    String dir = ModulePropertyManager.getPropertyManager(
			    ModulePropertyManager.TRANSPORTS_MODULE)
			    .getProperty(Environment.FTP_LOCALDIR,
				    System.getProperty("java.io.tmpdir"));

		    name = dir
			    + File.separator
			    + msg.getHeader().getCall().getMessageID()
				    .toString()
			    + ((FTPEpr) destination).getPostDirectory();
		}
		catch (URISyntaxException ex)
		{
		    throw new CourierException(ex);
		}
	    }
	    else if (destination instanceof FileEpr)
	    {
		try
		{
		    type = Environment.Transports.File;

		    name = ((FileEpr) destination).getURL()
			    + File.separator
			    + msg.getHeader().getCall().getMessageID()
				    .toString()
			    + ((FileEpr) destination).getInputSuffix();
		}
		catch (MalformedURLException ex)
		{
		    throw new CourierException(ex);
		}
		catch (URISyntaxException ex)
		{
		    throw new CourierException(ex);
		}
	    }
	    else if (destination instanceof JMSEpr)
	    {
		type = Environment.Transports.JMS;

		name = destination.getAddr().toString();
	    }
	    else if (destination instanceof JDBCEpr)
	    {
		type = Environment.Transports.SQL;

		name = destination.getAddr().toString();
	    }
	    else if (destination instanceof HibernateEpr)
	    {
		type = Environment.Transports.Hibernate;
		name = destination.getAddr().toString();
	    }
	    else
	    {
		type = null;
		name = null;
	    }
	}
	else
	{
	    type = null;
	    name = null;
	}

	final Properties props = msg.getProperties();
	setProperty(props, Environment.TRANSPORT_TYPE, type);
	setProperty(props, Environment.MESSAGE_SOURCE, name);
	props.setProperty(Environment.MESSAGE_ENTRY_TIME, Calendar
		.getInstance().getTime().toString());
	return msg;
    }

    public Message onInput (Message msg, Map<String, Object> params) throws CourierException
    {
	msg.getProperties().setProperty(Environment.MESSAGE_EXIT_TIME,
		Calendar.getInstance().getTime().toString());

	return msg;
    }

    private void setProperty (final Properties props, final String name,
	    final Object value)
    {
	if (value == null)
	{
	    props.remove(name);
	}
	else
	{
	    props.setProperty(name, value);
	}
    }
}
