/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */
package org.jboss.soa.esb.common;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.URL;

import javax.transaction.Status;
import javax.transaction.Synchronization;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.system.ServiceMBeanSupport;
import org.jboss.system.server.ServerConfig;
import org.jboss.system.server.ServerConfigLocator;
import org.jboss.tm.TransactionManagerLocator;

/**
 * This MBean wraps the configuration to allow ServiceBindingManager
 * to alter the configuration.
 */
public class JBossESBPropertyService extends ServiceMBeanSupport implements JBossESBPropertyServiceMBean
{
    /**
     * The name of the ESB property file.
     */
    private static final String ESB_PROPERTY_FILE = "jbossesb-properties.xml";
    
    /**
     * The current property file name.
     */
    private String propertyFile ;
    
    /**
     * Get the property file.
     * @return The name of the property file being used.
     */
    public String getPropertyFile()
    {
        return propertyFile ;
    }
    
    /**
     * Set the property file.
     * @param propertyFile The name of the property file being used.
     */
    public void setPropertyFile(final String propertyFile)
    {
        this.propertyFile = propertyFile ;
    }
    
    /**
     * Create the property service.
     */
    @Override
    protected void createService()
        throws Exception
    {
        if (propertyFile != null)
        {
            final File baseFile = new File(this.propertyFile) ;
            final InputStream xmlPropertyIS ;
            if (!baseFile.isAbsolute()) {
                final URL resourceURL = Thread.currentThread().getContextClassLoader().getResource(propertyFile) ;
                xmlPropertyIS = resourceURL.openStream() ;
            } else {
                xmlPropertyIS = new FileInputStream(baseFile) ;
            }
            
            try {
                final ServerConfig serverConfig = ServerConfigLocator.locate() ;
                final File dataDir = serverConfig.getServerDataDir() ;
                final File esbPropertyFile = new File(dataDir, ESB_PROPERTY_FILE) ;
                
                final FileOutputStream esbPropertyOS = new FileOutputStream(esbPropertyFile) ;
                try {
                    XMLHelper.replaceSystemProperties(XMLHelper.getXMLStreamReader(xmlPropertyIS),
                        XMLHelper.getXMLStreamWriter(esbPropertyOS)) ;
                } finally {
                    esbPropertyOS.close() ;
                }
                
                System.setProperty(Environment.PROPERTIES_FILE, "abs://" + esbPropertyFile.getAbsolutePath());
            } finally {
                xmlPropertyIS.close() ;
            }
        }
        // We also setup the transaction strategy here
        TransactionStrategy.setTransactionStrategy(new JTATransactionStrategy()) ;
    }
    
    /**
     * The JTA transaction strategy used in the application server.
     * @author kevin
     */
    private static class JTATransactionStrategy extends TransactionStrategy
    {
        /**
         * The logger for this class.
         */
        private static final Logger LOGGER = Logger.getLogger(JTATransactionStrategy.class) ;
        
        /**
         * The transaction manager.
         */
        private final TransactionManager tm ;
        
        /**
         * Construct the JTA transaction strategy.
         */
        JTATransactionStrategy()
        {
            tm = TransactionManagerLocator.getInstance().locate() ;
        }
        
        /**
         * Begin a transaction on the current thread.
         * @throws TransactionStrategyException
         */
        public void begin()
            throws TransactionStrategyException
        {
            try
            {
                tm.begin() ;
                LOGGER.debug("Transaction started on current thread") ;
            }
            catch (final Throwable th)
            {
                LOGGER.debug("Failed to start transaction on current thread", th) ;
                throw new TransactionStrategyException("Failed to begin transaction on current thread", th) ;
            }
        }
        
        /**
         * Terminate the transaction on the current thread.
         * If the transaction has been marked for rollback then it
         * will be rolled back, otherwise it will be committed.
         * @throws TransactionStrategyException
         */
        public void terminate()
            throws TransactionStrategyException
        {
            try
            {
                final int status = tm.getStatus() ;
                switch(status)
                {
                    case Status.STATUS_ACTIVE:
                        LOGGER.debug("Committing transaction on current thread") ;
                        tm.commit() ;
                        break ;
                    case Status.STATUS_MARKED_ROLLBACK:
                        LOGGER.debug("Rollback transaction on current thread") ;
                        tm.rollback();
                        break ;
                    default:
                        // Transaction is in some other state, just disassociate
                        if (LOGGER.isDebugEnabled())
                        {
                            LOGGER.debug("Suspending transaction on current thread, status: " + status) ;
                        }
                        tm.suspend() ;
                }
            }
            catch (final Throwable th)
            {
                LOGGER.debug("Failed to terminate transaction on current thread", th) ;
                throw new TransactionStrategyException("Failed to terminate transaction on current thread", th) ;
            }
        }
        
        /**
         * Mark the current transaction for rollback.
         * @throws TransactionStrategyException
         */
        public void rollbackOnly()
            throws TransactionStrategyException
        {
            try
            {
                tm.setRollbackOnly();
            }
            catch (final Throwable th)
            {
                throw new TransactionStrategyException("Failed to mark the transaction on current thread for rollback", th) ;
            }
        }
        
        /**
         * Get a handle on the currently associated transaction (or null).
         * @throws TransactionStrategyException
         */
        
        public Object getTransaction () throws TransactionStrategyException
        {
        	try
        	{
        		return tm.getTransaction();
        	}
        	catch (final Throwable th)
        	{
        		throw new TransactionStrategyException("Problem when trying to getTransaction: ",th);
        	}
        }
        
        /**
         * Suspend the current thread-to-transaction association.
         * 
         * @return the associated transaction, or null.
         * @throws TransactionStrategyException
         */
        public Object suspend () throws TransactionStrategyException
        {
        	try
        	{
        		return tm.suspend();
        	}
        	catch (final Throwable th)
        	{
        		throw new TransactionStrategyException("Problem when trying to suspend transaction: ",th);
        	}
        }
        
        /**
         * Associated the transaction with the current thread.
         * @param tx
         * @throws TransactionStrategyException
         */
        public void resume (Object tx) throws TransactionStrategyException
        {
        	try
        	{
        		tm.resume((Transaction) tx);
        	}
        	catch (final Throwable th)
        	{
        		throw new TransactionStrategyException("Problem when trying to resume transaction: ",th);
        	}
        }
        
        /**
         * Add a synchronization to the current transaction.
         * @param sync
         * @throws TransactionStrategyException
         */
        public void registerSynchronization (Synchronization sync) throws TransactionStrategyException
        {
            try
            {
        	tm.getTransaction().registerSynchronization(sync);
            }
            catch (final Throwable th)
            {
        	throw new TransactionStrategyException("Problem when registering synchronization: ", th);
            }
        }
        /**
         * Is the currently associated transaction active?
         * @return
         * @throws TransactionStrategyException
         */
        public boolean isActive () throws TransactionStrategyException
        {
        	try
        	{
        		if (tm.getStatus() == Status.STATUS_ACTIVE)
        			return true;
        		else
        			return false;
        	}
        	catch (final Throwable th)
        	{
        		throw new TransactionStrategyException("Problem when trying to get transaction status: ",th);
        	}
        }
        
        public String toString ()
        {
        	return "JTATransactionStrategy";
        }
    }
}
