/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.http.configurators;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.Credentials;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.http.Configurator;

import java.util.Properties;
import java.net.URI;

/**
 * HTTP BASIC Authentication Configurator.
 * <p/>
 * Properties:
 * <ul>
 *      <li><b>username</b>: See {@link UsernamePasswordCredentials}. (Required)</li>
 *      <li><b>password</b>: See {@link UsernamePasswordCredentials}. (Required)</li>
 *      <li><b>authscope-host</b>: See {@link AuthScope}. (Required)</li>
 *      <li><b>authscope-port</b>: See {@link AuthScope}. (Required)</li>
 *      <li><b>authscope-realm</b>: See {@link AuthScope}. (Optional)</li>
 * </ul>
 * <p/>
 * See <a href="http://jakarta.apache.org/commons/httpclient/authentication.html">HttpClient Authentication Guide</a>.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class AuthBASIC extends Configurator {

    public void configure(HttpClient httpClient, Properties properties) throws ConfigurationException {
        URI targetURI = getTargetURI(properties, false);
        String username = properties.getProperty("auth-username");
        String password = properties.getProperty("auth-password");
        String authScopeHost;
        String authScopePort;
        String authScopeRealm = properties.getProperty("authscope-realm");

        if(targetURI != null) {
            authScopeHost = properties.getProperty("authscope-host", targetURI.getHost());
            authScopePort = properties.getProperty("authscope-port", Integer.toString(targetURI.getPort()));
        } else {
            authScopeHost = properties.getProperty("authscope-host");
            authScopePort = properties.getProperty("authscope-port");
        }

        assertPropertySetAndNotBlank(username, "auth-username");
        assertPropertySetAndNotBlank(password, "auth-password");
        assertPropertySetAndNotBlank(authScopeHost, "authscope-host");
        assertPropertyIsInteger(authScopePort, "authscope-port");

        Credentials creds = new UsernamePasswordCredentials(username, password);
        AuthScope authScope;

        if(authScopeRealm != null && !authScopeRealm.trim().equals("")) {
            authScope = new AuthScope(authScopeHost, Integer.parseInt(authScopePort), authScopeRealm);
        } else {
            authScope = new AuthScope(authScopeHost, Integer.parseInt(authScopePort));
        }

        httpClient.getState().setCredentials(authScope, creds);
    }
}
