/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.List;

import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycle;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.parameters.ParamRepositoryException;
import org.jboss.soa.esb.parameters.ParamRepositoryFactory;
import org.jboss.soa.esb.util.ClassUtil;
import org.xml.sax.SAXException;

/**
 * Lifecycle helper methods
 * @author kevin
 */
public class LifecycleUtil
{
    /**
     * Default constructor.
     */
    private LifecycleUtil()
    {
    }
    
    /**
     * Get the managed listeners from the specified parameter entry in the repository.
     * @param param The listener parameter.
     * @return The managed listeners.
     * 
     * @throws ParamRepositoryException For errors associated with accessing the parameter repository.
     * @throws SAXException For errors associated with parsing the configuration.
     * @throws ManagedLifecycleException For errors associated with the managed instance.
     */
    public static List<ManagedLifecycle> getListeners(final String param)
        throws ParamRepositoryException, SAXException, ManagedLifecycleException
    {
        return getListeners(getConfigTree(param)) ;
    }
    
    /**
     * Get the managed listeners from the specified config tree.
     * @param tree The config tree.
     * @return The managed listeners.
     * 
     * @throws ManagedLifecycleException For errors associated with the managed instance.
     */
    public static List<ManagedLifecycle> getListeners(final ConfigTree tree)
        throws ManagedLifecycleException
    {
        return getManagedInstances(tree, ListenerTagNames.LISTENER_CLASS_TAG) ;
    }
    
    /**
     * Get the managed gateways from the specified parameter entry in the repository.
     * @param param The gateway parameter.
     * @return The managed gateways.
     * 
     * @throws ParamRepositoryException For errors associated with accessing the parameter repository.
     * @throws SAXException For errors associated with parsing the configuration.
     * @throws ManagedLifecycleException For errors associated with the managed instance.
     */
    public static List<ManagedLifecycle> getGateways(final String param)
        throws ParamRepositoryException, SAXException, ManagedLifecycleException
    {
        return getGateways(getConfigTree(param)) ;
    }
    
    /**
     * Get the managed gateways from the specified config tree.
     * @param tree The config tree.
     * @return The managed gateways.
     * 
     * @throws ManagedLifecycleException For errors associated with the managed instance.
     */
    public static List<ManagedLifecycle> getGateways(final ConfigTree tree)
        throws ManagedLifecycleException
    {
        return getManagedInstances(tree, ListenerTagNames.GATEWAY_CLASS_TAG) ;
    }
    
    /**
     * Get the managed instances from the specified config tree.
     * @param tree The config tree.
     * @return The managed instances.
     * 
     * @throws ParamRepositoryException For errors associated with accessing the parameter repository.
     * @throws SAXException For errors associated with parsing the configuration.
     * @throws ManagedLifecycleException For errors associated with the managed instance.
     */
    private static List<ManagedLifecycle> getManagedInstances(final ConfigTree tree, final String classTag)
        throws ManagedLifecycleException
    {
        final ArrayList<ManagedLifecycle> instances = new ArrayList<ManagedLifecycle>() ;
        
        for(ConfigTree child : tree.getAllChildren())
        {
            final String classname = child.getAttribute(classTag) ;
            if (classname != null)
            {
                final Class instanceClass ;
                try
                {
                    instanceClass = ClassUtil.forName(classname, LifecycleUtil.class) ;
                }
                catch (final ClassNotFoundException cnfe)
                {
                    throw new ManagedLifecycleException("Could not locate managed instance " + classname) ;
                }
                
                if (!ManagedLifecycle.class.isAssignableFrom(instanceClass))
                {
                    throw new ManagedLifecycleException("Managed instance " + classname + " does not implement ManagedLifecycle") ;
                }
                
                final Constructor constructor ;
                try
                {
                    constructor = instanceClass.getConstructor(new Class[] {ConfigTree.class}) ;
                }
                catch (final NoSuchMethodException nsme)
                {
                    throw new ManagedLifecycleException("Managed instance [" + classname +
                            "] does not have correct constructor.  This class must contain a public constructor of the form 'public " +
                            instanceClass.getSimpleName() + "(" + ConfigTree.class.getName() + " config);'");
                }
                final ManagedLifecycle instance ;
                try
                {
                    instance = (ManagedLifecycle)constructor.newInstance(child) ;
                }
                catch (final Exception ex)
                {
                    throw new ManagedLifecycleException("Unexpected exception while instantiating managed instance", ex) ;
                }
                instances.add(instance) ;
            }
        }
        
        return instances ;
    }
    
    /**
     * Get the config tree from the specified parameter entry in the repository.
     * @param param The parameter.
     * @return The config tree.
     * 
     * @throws ParamRepositoryException For errors associated with accessing the parameter repository.
     * @throws SAXException For errors associated with parsing the configuration.
     */
    public static ConfigTree getConfigTree(final String param)
        throws ParamRepositoryException, SAXException
    {
        final String xml = ParamRepositoryFactory.getInstance().get(param) ;
        final ConfigTree tree = ConfigTree.fromXml(xml) ;
        
        tree.setAttribute("configSource", "param-repository:" + param) ;
        return tree ;
    }
}
