/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.config.mappers;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.listeners.config.Generator;
import org.jboss.soa.esb.listeners.config.ScheduleProviderFactory;
import org.jboss.soa.esb.listeners.config.xbeanmodel.AbstractScheduledListener;
import org.w3c.dom.Element;

/**
 * Schedule Mapper.
 * <p/>
 * Maps the scheduling info from the Schedule based listener config, to the
 * ConfigTree config model.
 *
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ScheduleMapper {

    private static Logger logger = Logger.getLogger(ScheduleMapper.class);

    public static void map(Element listenerConfig, AbstractScheduledListener scheduledListener) throws ConfigurationException {
        String scheduleIdRef = scheduledListener.getScheduleidref();
        Element domElement = (Element) scheduledListener.getDomNode();
        int frequency = scheduledListener.getScheduleFrequency();
        int pollFrequency = scheduledListener.getPollFrequencySeconds();
        int pollLatency = getPollLatencySeconds(scheduledListener);

        if(scheduleIdRef != null) {
            listenerConfig.setAttribute(ScheduleProviderFactory.ATTR_SCHEDULE_ID_REF, scheduleIdRef);
            if(domElement.hasAttribute("poll-frequency-seconds") || domElement.hasAttribute("schedule-frequency") ) {
                logger.warn("Schedule Listener '" + listenerConfig.getTagName() + "' defines both 'scheduleidref' and frequency attributes.  Using the 'scheduleidref'.");
            }
        } else if(frequency > -1) {
            listenerConfig.setAttribute(ScheduleProviderFactory.ATTR_FREQUENCY, Integer.toString(frequency));
        } else if(pollFrequency > -1) {
            listenerConfig.setAttribute(ScheduleProviderFactory.ATTR_FREQUENCY, Integer.toString(pollFrequency));
        } else if(pollLatency > -1) {
            listenerConfig.setAttribute(ScheduleProviderFactory.ATTR_FREQUENCY, Integer.toString(pollLatency));
        } else {
            listenerConfig.setAttribute(ScheduleProviderFactory.ATTR_FREQUENCY, "10");
        }

        if(domElement.hasAttribute("poll-frequency-seconds")) {
            logger.warn("Attrubute 'poll-frequency-seconds' is DEPRECATED.  Please change your configuration to use 'schedule-frequency'.");
        }
        if(pollLatency > -1) {
            logger.warn("Property 'pollLatencySeconds' is DEPRECATED.  Please change your configuration to use 'schedule-frequency'.");
        }
    }

    private static int getPollLatencySeconds(AbstractScheduledListener scheduledListener) {
        String value = Generator.XMLBeansModel.getProperty(scheduledListener.getPropertyList(), "pollLatencySeconds", "-1");

        try {
            return Integer.parseInt(value);
        } catch (NumberFormatException e) {
            return -1;
        }
    }
}