/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners.gateway;

import java.io.File;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.gateway.remotestrategies.ReadOnlyRemoteFileSystemStrategy;
import org.jboss.soa.esb.listeners.gateway.remotestrategies.RemoteFileSystemStrategy;
import org.jboss.soa.esb.listeners.gateway.remotestrategies.RemoteFileSystemStrategyException;
import org.jboss.soa.esb.listeners.gateway.remotestrategies.cache.DeleteOnEvictTreeCacheListener;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * This class extends {@link RemoteGatewayListener} and is intended to be used
 * when the remote file system does not support write operations.
 * <p>
 * 
 * <pre>
 * Sample Listener Configuration:
 * &lt;ftp-listener name="FtpGateway"
 *  busidref="helloFTPChannel"
 *   maxThreads="1"
 *    is-gateway="true"&gt;
 *     &lt;property name="pollLatencySeconds" value="5"/&gt; &lt;!-- Interval to poll the remote file system --&gt;
 *     &lt;property name="remoteFileSystemStrategy-configFile" value="/ftpfile-cache-config.xml"/&gt; &lt;!-- Optional--&gt;
 *     &lt;property name="remoteFileSystemStrategy-cacheListener" value=""org.jboss.soa.esb.listeners.gateway.remotestrategies.cache.DeleteOnEvictTreeCacheListener"/&gt; &lt;!-- Optional--&gt;
 *     &lt;property name="remoteFileSystemStrategy-class" value="org.my.MyRemoteGatewayStrategy"/&gt; &lt;!-- Optional--&gt;
 *&lt;/ftp-listener&gt;
 * </pre>
 * 
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>				
 *
 */
public class ReadOnlyRemoteGatewayListener extends RemoteGatewayListener
{
	/**
	 * Property to specify the {@link RemoteFileSystemStrategy} implementation
	 */
	public static final String REMOTE_FILE_SYSTEM_STRATEGY_CLASS = "remoteFileSystemStrategy-class";
	/**
	 * Property to specify a configuration file for {@link #REMOTE_FILE_SYSTEM_STRATEGY_CLASS} 
	 */
	public static final String REMOTE_FILE_SYSTEM_STRATEGY_CONFIG_FILE = "remoteFileSystemStrategy-configFile";
	/**
	 * Property to specify if a the TreeCacheListener should be used. See {@link DeleteOnEvictTreeCacheListener} 
	 */
	public static final String REMOTE_FILE_SYSTEM_STRATEGY_CACHE_LISTENER = "remoteFileSystemStrategy-cacheListener";
	/**
	 * A strategy for handling call related to write operations.
	 */
	private RemoteFileSystemStrategy remotefileSystemStrategy;
    /**
     * serial version uid for this class
     */
    private static final long serialVersionUID = 8505559166811233906L;
    
    public ReadOnlyRemoteGatewayListener(ConfigTree config) throws ConfigurationException, RegistryException, GatewayException {
		super(config);
		createRemoteFileSystemStrategy();
	}

	/**
	 * Delegates to {@link RemoteFileSystemStrategy#deleteFile(File)}
	 */
	@Override
	public boolean deleteFile( File file ) throws GatewayException
	{
		try
		{
			return remotefileSystemStrategy.deleteFile( file );
		} 
		catch (RemoteFileSystemStrategyException e)
		{
			throw new GatewayException( e );
		}
	}

	/**
	 * Delegates to {@link RemoteFileSystemStrategy#filterFileList(File[])}
	 */
	@Override
	File[] getFileList() throws GatewayException
	{
		try
		{
			return remotefileSystemStrategy.filterFileList( super.getFileList() );
		} 
		catch (RemoteFileSystemStrategyException e)
		{
			throw new GatewayException( e );
		}
	}

	/**
	 * Delegates to {@link RemoteFileSystemStrategy#renameFile(File, File)}
	 */
	@Override
	boolean renameFile(File from, File to) throws GatewayException
	{
		try
		{
			return remotefileSystemStrategy.renameFile( from, to );
		} 
		catch (RemoteFileSystemStrategyException e)
		{
			throw new GatewayException( e );
		}
	}
	
	/**
	 * Delegates to {@link RemoteFileSystemStrategy#getWorkFileName(File, String)}
	 */
	@Override
	protected File getWorkFileName(File file, String suffix )
	{
		return remotefileSystemStrategy.getWorkFileName( file, suffix );
	}
	
	/**
	 * Set the {@link RemoteFileSystemStrategy } to be used
	 * @param remotefileSystemStrategy
	 */
	public void setRemotefileSystemStrategy( RemoteFileSystemStrategy remotefileSystemStrategy )
	{
		this.remotefileSystemStrategy = remotefileSystemStrategy;
	}
	
	protected <T> T getNewInstanceOf( String className ) throws ConfigurationException
	{
		if ( className == null )
			throw new IllegalArgumentException ( "className argument must not be null.");
		
		T instance = null;
		try
		{
			@SuppressWarnings ("unchecked")
            Class<T> type = ClassUtil.forName( className , getClass());
            instance = (T) type.newInstance();
		}
		catch (Exception ex)
		{
			_logger.error(ex);
			throw new ConfigurationException(ex);
		}
		return instance;
	}

	/**
	 * Stop the service and also after that stop the 
	 * remotefileSystemStrategy in use
	 */
	protected void doStop() throws ManagedLifecycleException
	{
		remotefileSystemStrategy.stop();
		super.doStop();
	}
	
	/**
	 * Destroys the service and also after that stop the 
	 * remotefileSystemStrategy in use
	 */
    protected void doThreadedDestroy() throws ManagedLifecycleException
    {
		remotefileSystemStrategy.destroy();
		super.doDestroy();
    }
	
	private void createRemoteFileSystemStrategy() throws ConfigurationException
	{
		String strategyClassName = config.getAttribute( REMOTE_FILE_SYSTEM_STRATEGY_CLASS );
		RemoteFileSystemStrategy strategy = null;
		
		if ( strategyClassName != null )
			strategy = getNewInstanceOf( strategyClassName );
		
		remotefileSystemStrategy = strategy != null ? strategy : new ReadOnlyRemoteFileSystemStrategy();

		String configFile = config.getAttribute( REMOTE_FILE_SYSTEM_STRATEGY_CONFIG_FILE );
		try
		{
			remotefileSystemStrategy.init( configFile );
				
			if ( remotefileSystemStrategy instanceof ReadOnlyRemoteFileSystemStrategy )
			{
				ReadOnlyRemoteFileSystemStrategy rm = (ReadOnlyRemoteFileSystemStrategy) remotefileSystemStrategy;
				String cacheListener = config.getAttribute( REMOTE_FILE_SYSTEM_STRATEGY_CACHE_LISTENER );
				
				if ( cacheListener != null )
				{
		           Class cacheListenerClass = ClassUtil.forName( cacheListener , getClass() );
		           rm.setCacheListener( cacheListenerClass );
				}
			}
		}
		catch( RemoteFileSystemStrategyException e )
		{
			_logger.error( "Could not configure the remoteFileSystemStrategy.", e );
			throw new ConfigurationException(e);
		} 
		catch (ClassNotFoundException e)
		{
			_logger.error( "ClassNotFound while trying to create the the remoteFileSystemStrategy.", e );
			throw new ConfigurationException(e);
		}
	}

}
