/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.notification;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URISyntaxException;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.jboss.soa.esb.addressing.eprs.FTPEpr;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.util.RemoteFileSystem;
import org.jboss.soa.esb.util.RemoteFileSystemException;
import org.jboss.soa.esb.util.RemoteFileSystemFactory;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;

/**
 * Sends a message to an outgoing FTP server. The outgoing filename can have
 * values of message properties injected into it to make it unique. The
 * notification-details property looks something like:
 * 
 * <code>
 *		<NotificationList type="OK" xmlns="http://anonsvn.labs.jboss.com/labs/jbossesb/trunk/product/etc/schemas/xml/jbossesb-1.0.1.xsd">
 *			<target class="NotifyFTP">
 *				<ftp URL="ftp://username:pwd@server.com/remote/dir" filename="{jbossesb.message.id}.txt"/>
 *			</target>
 *		</NotificationList>
 * </code>
 * 
 * @author <a href="mailto:rex.sheridan@sapience360.com">Rex Sheridan</a>
 */
public class NotifyFTP extends NotificationTarget {

	/** Configuration entries */
	private ConfigTree config;

	/** Used to get access to remote filesystem */
	private FTPEpr epr;

	/** Filename created on server */
	private String fileName;

	/** FTP child tag name */
	public static final String CHILD_FTP = "ftp";

	/** Filename attribute */
	public static final String ATTR_FILENAME = "filename";

	/** Base filename for temp file */
	public static final String TEMP_FILE_BASE = "jbossesb-NotifyFTP";
    private MessagePayloadProxy payloadProxy;

    /**
	 * Create an outgoing FTP notification based on the given configuration.
	 * 
	 * @param config
	 */
	public NotifyFTP(ConfigTree config) {
		this.config = config;
        payloadProxy = new MessagePayloadProxy(config,
                                               new String[] {BytesBody.BYTES_LOCATION},
                                               new String[] {BytesBody.BYTES_LOCATION});
	}

	/**
	 * Get the ConfigTree for the 'ftp' element.
	 * 
	 * @return ConfigTree
	 */
	protected ConfigTree getFtpConfig() throws NotificationException {
		ConfigTree[] ftps = config.getChildren(CHILD_FTP);
		if (ftps.length != 1) {
			throw new NotificationException("NotifyFTP requires exactly one 'ftp' element.");
		}
		return ftps[0];
	}

	/**
	 * Lazy loads filename from config tree.
	 * 
	 * @return String
	 */
	protected String getFileName(Message message) throws NotificationException {
		if (fileName == null) {
			ConfigTree ftpConfig = getFtpConfig();
			fileName = ftpConfig.getAttribute(ATTR_FILENAME);
			if (StringUtils.isEmpty(fileName)) {
				throw new NotificationException("NotifyFTP: Filename attribute is required.");
			}
			fileName = replaceArguments(fileName, message);
		}
		return fileName;
	}

	/**
	 * Replaces any arguments in the form '{prop.name}' with corresponding
	 * message property values.
	 * 
	 * @param value
	 * @param message
	 * @return String
	 */
	protected String replaceArguments(String value, Message message) {
		String current = value;
		String replaced = replaceArgument(value, message);
		while (!current.equals(replaced)) {
			current = replaced;
			replaced = replaceArgument(current, message);
		}
		return replaced;
	}

	/**
	 * Look for arguments in the form '{prop.name}' and replace them with
	 * corresponding message properties.
	 * 
	 * @param value
	 * @param message
	 * @return String
	 */
	protected String replaceArgument(String value, Message message) {
		int startIndex = value.indexOf('{');
		if (startIndex == -1) {
			return value;
		}
		int endIndex = value.indexOf('}');
		if (endIndex == -1) {
			return value;
		}
		String propName = value.substring(startIndex + 1, endIndex);
		Object propValue = message.getProperties().getProperty(propName);
		if (propValue == null) {
			return value;
		}
		return value.substring(0, startIndex) + propValue + value.substring(endIndex + 1);
	}

	/**
	 * Builds an FTP EPR from the configutation data.
	 * 
	 * @return FTPEpr
	 */
	protected FTPEpr getFtpEpr() throws NotificationException {
		if (epr == null) {
			ConfigTree ftpConfig = getFtpConfig();
			String url = ftpConfig.getAttribute(FTPEpr.URL_TAG);
			try {
				epr = new FTPEpr(url);
			} catch (URISyntaxException e) {
				throw new NotificationException(e);
			}
		}
		return epr;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.notification.NotificationTarget#sendNotification(org.jboss.soa.esb.message.Message)
	 */
	public void sendNotification(Message message) throws NotificationException {
		FileOutputStream stream = null;
		File tmpFile = null;
		RemoteFileSystem rfs = null;
		try {
			tmpFile = File.createTempFile(TEMP_FILE_BASE, null);
			stream = new FileOutputStream(tmpFile);
			
			Object payload = payloadProxy.getPayload(message);
			
			if (payload instanceof byte[])
				IOUtils.write((byte[]) payload, stream);
			else
				IOUtils.write(payload.toString(), stream);
			
            stream.close();
			rfs = RemoteFileSystemFactory.getRemoteFileSystem(getFtpEpr(), true);
			rfs.uploadFile(tmpFile, getFileName(message));
        } catch (MessageDeliverException e) {
            e.printStackTrace();
		} catch (RemoteFileSystemException e) {
			throw new NotificationException("Could not complete FTP notification", e);
		} catch (IOException e) {
			throw new NotificationException("Could not complete FTP notification", e);
		} finally {
			if (stream != null) {
				IOUtils.closeQuietly(stream);
			}
			if (tmpFile != null) {
				tmpFile.delete();
			}
			if (rfs != null) {
				rfs.quit();
			}
		}
	}
}