/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.couriers.tests;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.PrintStream;
import java.net.URI;
import java.util.UUID;

import junit.framework.Assert;
import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.eprs.FileEpr;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierFactory;
import org.jboss.soa.esb.couriers.CourierUtil;
import org.jboss.soa.esb.couriers.FaultMessageException;
import org.jboss.soa.esb.couriers.TwoWayCourier;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

import com.arjuna.common.util.propertyservice.PropertyManager;

/**
 * Tests for internal FileCourier class 
 * @author <a href="mailto:schifest@heuristica.com.ar">schifest@heuristica.com.ar</a>
 * @since Version 4.0
 *
 */
public class FileCourierUnitTest 
{
	private static Logger _logger		= Logger.getLogger(FileCourierUnitTest.class);
	private static final String TMP_DIR = System.getProperty("java.io.tmpdir");
	
	static File _tmpDir = new File(TMP_DIR);
	static String TEST_SUFFIX	= ".testMessage";
	static String ERROR_SUFFIX	= ".myErrorSuffix";
	static String WORK_SUFFIX	= ".inProcessMessage";
	static String DONE_SUFFIX	= ".doneMessage";
	
	@BeforeClass
	public static void setUp() throws Exception
	{
		_logger.debug("tmp directory = <"+_tmpDir+">");
		purgeStaleFiles();
		
		PropertyManager pm = ModulePropertyManager.getPropertyManager(ModulePropertyManager.FILTER_MODULE);
		pm.setProperty("org.jboss.soa.esb.filter.0", "org.jboss.internal.soa.esb.message.filter.MetaDataFilter");
	}
	
	@AfterClass
	public static void tearDown() throws Exception
	{
		purgeStaleFiles();
	}

	public static junit.framework.Test suite() {
		return new JUnit4TestAdapter(FileCourierUnitTest.class);
	}
	
	private static void purgeStaleFiles()
	{
		FileFilter ff = new FileFilter() 
		{
				public boolean accept(File file)
				{
					return 
						(  file.getName().endsWith(TEST_SUFFIX)
						 ||file.getName().endsWith(WORK_SUFFIX)
						 ||file.getName().endsWith(DONE_SUFFIX)
						 ||file.getName().endsWith(ERROR_SUFFIX)
						);
				}
			};
		for (File file:_tmpDir.listFiles(ff))
		{
			_logger.debug("delete of "+file.toString()+" = "+file.delete());
		}
		
	}

	@Test
	public void testMessageAttributes() throws Exception 
    {
		String contents = "This is the text that travels in the Message body";

		// toEpr for files must be a directory
		FileEpr toEpr = new FileEpr(_tmpDir.toURL());
		toEpr.setInputSuffix(TEST_SUFFIX);
		
		Message msg = MessageFactory.getInstance().getMessage();
		msg.getBody().add(contents.getBytes());

		Call call = new Call(toEpr);
		String uid = UUID.randomUUID().toString();
		call.setMessageID(new URI(uid));
		msg.getHeader().setCall(call);
		
		CourierUtil.deliverMessage(msg);
		
		String timeSent = (String) msg.getProperties().getProperty(Environment.MESSAGE_ENTRY_TIME);
		String source = (String) msg.getProperties().getProperty(Environment.MESSAGE_SOURCE);
		
		File theFile = new File(_tmpDir,uid+TEST_SUFFIX);
		Assert.assertTrue(theFile.exists());
		_logger.info("Message file "+theFile.toString()+" successfully created");
				
		FileEpr fromEpr = new FileEpr(toEpr.getURL());
		fromEpr.setInputSuffix(TEST_SUFFIX);
		fromEpr.setPostSuffix(DONE_SUFFIX);
		
		TwoWayCourier pickUp = CourierFactory.getPickupCourier(fromEpr);
		Message retrieved = pickUp.pickup(1000);
		Assert.assertFalse("Null message retrieved",null==retrieved);
		
		String back = new String((byte[]) retrieved.getBody().get());
		Assert.assertEquals(contents,back);
		_logger.info("Contents of retrieved msg equal original text <"+back+">");
		
		theFile = new File(_tmpDir,uid+TEST_SUFFIX+DONE_SUFFIX);
		Assert.assertTrue(theFile.exists());
		_logger.info("Retrieved message properly renamed to <"+theFile.toString()+">");
		
		Assert.assertEquals(timeSent, retrieved.getProperties().getProperty(Environment.MESSAGE_ENTRY_TIME));
		Assert.assertEquals(source, retrieved.getProperties().getProperty(Environment.MESSAGE_SOURCE));
		Assert.assertNotNull(retrieved.getProperties().getProperty(Environment.MESSAGE_EXIT_TIME));
		
		purgeStaleFiles();
    }
	
	@Test
	public void testFaultMessage () throws Exception 
    {
		// toEpr for files must be a directory
		FileEpr toEpr = new FileEpr(_tmpDir.toURL());
		toEpr.setInputSuffix(TEST_SUFFIX);
		
		Message msg = MessageFactory.getInstance().getMessage();
		String errorMessage = "Some failure occurred!";
		URI code = new URI("urn:errorcode");
		
		msg.getFault().setReason(errorMessage);
		msg.getFault().setCode(code);
		msg.getFault().setCause(new IOException());

		Call call = new Call(toEpr);
		String uid = UUID.randomUUID().toString();
		call.setMessageID(new URI(uid));
		msg.getHeader().setCall(call);
		
		CourierUtil.deliverMessage(msg);
		
		File theFile = new File(_tmpDir,uid+TEST_SUFFIX);
		Assert.assertTrue(theFile.exists());
		_logger.info("Message file "+theFile.toString()+" successfully created");
				
		FileEpr fromEpr = new FileEpr(toEpr.getURL());
		fromEpr.setInputSuffix(TEST_SUFFIX);
		fromEpr.setPostSuffix(DONE_SUFFIX);
		
		TwoWayCourier pickUp = CourierFactory.getPickupCourier(fromEpr);

		try
		{
		    Message retrieved = pickUp.pickup(1000);
		    _logger.info(retrieved);
		    Assert.fail();
		}
		catch (FaultMessageException ex)
		{
		    Assert.assertEquals(ex.getMessage(), errorMessage);
		    Assert.assertEquals(ex.getCode(), code);
		    Assert.assertEquals(ex.getCause() instanceof IOException, true);
		}
		catch (CourierException ex)
		{
		    Assert.fail();
		}
		
		purgeStaleFiles();
    }
	
	@Test
	public void testStoreRetrieve() throws Exception 
    {
		String contents = "This is the text that travels in the Message body";

		// toEpr for files must be a directory
		FileEpr toEpr = new FileEpr(_tmpDir.toURL());
		toEpr.setInputSuffix(TEST_SUFFIX);
		
		Message msg = MessageFactory.getInstance().getMessage();
		msg.getBody().add(contents.getBytes());

		Call call = new Call(toEpr);
		String uid = UUID.randomUUID().toString();
		call.setMessageID(new URI(uid));
		msg.getHeader().setCall(call);
		
		CourierUtil.deliverMessage(msg);
		File theFile = new File(_tmpDir,uid+TEST_SUFFIX);
		Assert.assertTrue(theFile.exists());
		_logger.info("Message file "+theFile.toString()+" successfully created");
				
		FileEpr fromEpr = new FileEpr(toEpr.getURL());
		fromEpr.setInputSuffix(TEST_SUFFIX);
		fromEpr.setPostSuffix(DONE_SUFFIX);
		
		TwoWayCourier pickUp = CourierFactory.getPickupCourier(fromEpr);
		Message retrieved = pickUp.pickup(1000);
		Assert.assertFalse("Null message retrieved",null==retrieved);
		
		String back = new String((byte[]) retrieved.getBody().get());
		Assert.assertEquals(contents,back);
		_logger.info("Contents of retrieved msg equal original text <"+back+">");
		
		theFile = new File(_tmpDir,uid+TEST_SUFFIX+DONE_SUFFIX);
		Assert.assertTrue(theFile.exists());
		_logger.info("Retrieved message properly renamed to <"+theFile.toString()+">");
		
		purgeStaleFiles();
    }

	@Test
	public void testStoreAndDelete() throws Exception 
    {
		String contents = "This is the text that travels in the Message body";

		// toEpr for files must be a directory
		FileEpr toEpr = new FileEpr(_tmpDir.toURL());
		toEpr.setInputSuffix(TEST_SUFFIX);
		
		Message msg = MessageFactory.getInstance().getMessage();
		msg.getBody().add(contents.getBytes());
		msg.getBody().add("exampleObj", new ExampleObject("hello", 1234));
		
		Call call = new Call(toEpr);
		final String uid = UUID.randomUUID().toString();
		call.setMessageID(new URI(uid));
		msg.getHeader().setCall(call);
		
		CourierUtil.deliverMessage(msg);
		File theFile = new File(_tmpDir,uid+TEST_SUFFIX);
		Assert.assertTrue(theFile.exists());
		_logger.info("Message file "+theFile.toString()+" successfully created");
				
		FileEpr fromEpr = new FileEpr(toEpr.getURL());
		fromEpr.setInputSuffix(TEST_SUFFIX);
		// setting postdelete to true will delete input Message files, if they were picked up
		fromEpr.setPostDelete(true);
		
		TwoWayCourier pickUp = CourierFactory.getPickupCourier(fromEpr);
		Message retrieved = pickUp.pickup(1000);
		Assert.assertFalse("Null message retrieved",null==retrieved);
		
		String back = new String((byte[]) retrieved.getBody().get());
		Assert.assertEquals(contents,back);
		_logger.info("Contents of retrieved msg equal original text <"+back+">");
		
		Object payload = retrieved.getBody().get("exampleObj");
		
		Assert.assertEquals(payload instanceof ExampleObject, true);
		
		ExampleObject payloadType = (ExampleObject) payload;
		
		Assert.assertEquals(payloadType.name, "hello");
		Assert.assertEquals(payloadType.value, 1234);
		
		FileFilter ff = new FileFilter()
		{ public boolean accept(File file)
			{return file.getName().startsWith(uid); }
		};
		File[] all = new File(TMP_DIR).listFiles(ff);
		Assert.assertTrue(null==all || all.length<1);
		_logger.info("Good message file properly deleted ");
		
		purgeStaleFiles();
    }

	@Test
	public void testBadMessageRename() throws Exception 
    {
		String uid = UUID.randomUUID().toString();
		File theFile = new File(_tmpDir,uid+TEST_SUFFIX);
		PrintStream out = new PrintStream(theFile);
		out.print("This is an invalid message");
		out.close();
		Assert.assertTrue(theFile.exists());
		_logger.info("Invalid Message file "+theFile.toString()+" successfully created");
				
		FileEpr fromEpr = new FileEpr(_tmpDir.toURL()) ;
		fromEpr.setInputSuffix(TEST_SUFFIX);
		fromEpr.setPostSuffix(DONE_SUFFIX);
		fromEpr.setErrorSuffix(ERROR_SUFFIX);
		// Don't delete - default is to delete files in error
		fromEpr.setErrorDelete(false);
		
		TwoWayCourier pickUp = CourierFactory.getPickupCourier(fromEpr);
		Message retrieved = pickUp.pickup(1000);
		Assert.assertTrue("Null message retrieved",null==retrieved);
		
		theFile = new File(_tmpDir,uid+TEST_SUFFIX+ERROR_SUFFIX);
		Assert.assertTrue(theFile.exists());
		_logger.info("Message file properly renamed to "+theFile.toString());
		
		purgeStaleFiles();
    }

	@Test
	public void testBadMessageDelete() throws Exception 
    {
		final String uid = UUID.randomUUID().toString();
		File theFile = new File(_tmpDir,uid+TEST_SUFFIX);
		PrintStream out = new PrintStream(theFile);
		out.print("This is an invalid message");
		out.close();
		Assert.assertTrue(theFile.exists());
		_logger.info("Invalid Message file "+theFile.toString()+" successfully created");
				
		FileEpr fromEpr = new FileEpr(_tmpDir.toURL());
		fromEpr.setInputSuffix(TEST_SUFFIX);
		fromEpr.setPostSuffix(DONE_SUFFIX);
		fromEpr.setErrorSuffix(ERROR_SUFFIX);
		// Not using setErrorDelete(false) should delete files in error
		
		TwoWayCourier pickUp = CourierFactory.getPickupCourier(fromEpr);
		Message retrieved = pickUp.pickup(1000);
		Assert.assertTrue("Null message retrieved",null==retrieved);
		
		FileFilter ff = new FileFilter()
		{ public boolean accept(File file)
			{return file.getName().startsWith(uid); }
		};
		File[] all = new File(TMP_DIR).listFiles(ff);
		Assert.assertTrue(null==all || all.length<1);
		_logger.info("Bad message file properly deleted ");
		
		purgeStaleFiles();
    }

}
