/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners;

import java.io.File;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.Statement;
import java.util.Properties;

import junit.framework.TestCase;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.eprs.FileEpr;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.RegistryUtil;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.testutils.FileUtil;
import org.jboss.soa.esb.testutils.HsqldbUtil;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;

public class ServiceInvokerUnitTest extends TestCase
{
	private Logger log = Logger.getLogger( ServiceInvokerUnitTest.class );

	protected final void setup()
	{
		try
		{
			TestEnvironmentUtil.setESBPropertiesFileToUse("product",
					"../product");
			// Set the juddi properties file in System so juddi will pick it up
			// later and use the test values.
			String juddiPropertiesFile = "/org/jboss/soa/esb/listeners/juddi-unittest.properties";
			System.setProperty("juddi.propertiesFile", juddiPropertiesFile);
			// Read this properties file to get the db connection string
			Properties props = new Properties();
			InputStream inStream = Class.class
					.getResourceAsStream(juddiPropertiesFile);

			props.load(inStream);
			mDbDriver = props.getProperty("juddi.jdbcDriver");
			mDbUrl = props.getProperty("juddi.jdbcUrl");
			mDbUsername = props.getProperty("juddi.jdbcUsername");
			mDbPassword = props.getProperty("juddi.jdbcPassword");

			String database = "not tested yet";
			if ("org.hsqldb.jdbcDriver".equals(mDbDriver))
			{
				database = "hsqldb";
				// Bring up hsql on default port 9001
				HsqldbUtil.startHsqldb(TestEnvironmentUtil.getUserDir(
						"product", "../product")
						+ "/build/hsqltestdb", "juddi");
			}
			else if ("com.mysql.jdbc.Driver".equals(mDbDriver))
			{
				database = "mysql";
			} // add and test your own database..

			// Get the registry-schema create scripts
			String sqlDir = TestEnvironmentUtil.getUserDir("product",
					"../product")
					+ "/install/jUDDI-registry/sql/" + database + "/";
			// Drop what is there now, if exists. We want to start fresh.
			String sqlDropCmd = FileUtil.readTextFile(new File(sqlDir
					+ "drop_database.sql")).replaceAll("\\$\\{prefix}", "");
            String resource = "juddi-sql/" + database + "/create_database.sql";
            InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(resource);
            String sqlCreateCmd    = FileUtil.readStream(is).trim().replaceAll("\\$\\{prefix}", "");
			String sqlInsertPubCmd = FileUtil.readTextFile(new File(sqlDir
					+ "import.sql")).trim().replaceAll("\\$\\{prefix}", "");

			try
			{
				Class.forName(mDbDriver);
			}
			catch (Exception e)
			{
				log.error("ERROR: failed to load " + database
						+ " JDBC driver.", e);
				return;
			}
			con = DriverManager.getConnection(mDbUrl, mDbUsername, mDbPassword);
			Statement stmnt = con.createStatement();
			stmnt.execute(sqlDropCmd);
			stmnt.execute(sqlCreateCmd);
			stmnt.execute(sqlInsertPubCmd);
			stmnt.close();
		}
		catch (Throwable e)
		{
			log.error("We should stop testing, since we don't have a db.", e);
			assertTrue(false);
		}
	}

	protected final void tearDown()
	{
		try
		{
			Thread.sleep(1000);
			Statement stmnt = con.createStatement();

			stmnt.execute("SHUTDOWN");
			stmnt.close();

			con.close();
		}
		catch (Exception ex)
		{
			log.error(ex);
		}
	}

	public void testInvalidParameters() throws Exception
	{
		setup();

		Message message = MessageFactory.getInstance().getMessage();

		try
		{
            org.jboss.soa.esb.client.ServiceInvoker invoker = new org.jboss.soa.esb.client.ServiceInvoker("foo", "bar");
            invoker.deliverAsync(message);
			fail();
		}
		catch (MessageDeliverException ex)
		{
		}

		final File tmpDir = new File(System.getProperty("java.io.tmpdir"));
		final String tmpDirForm = tmpDir.toURL().toExternalForm();
		FileEpr epr = new FileEpr(tmpDirForm);

		epr.setInputSuffix(".testFile");
		epr.setPostDelete(true);
		epr.setPostSuffix(".unitProcessedOK");

		ConfigTree tree = new ConfigTree("test");

		tree.setAttribute(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG,
				"eprmanager");
		tree.setAttribute(ListenerTagNames.SERVICE_NAME_TAG, "qwerty");

		RegistryUtil.register(tree, epr);

		try
		{
            ServiceInvoker invoker = new org.jboss.soa.esb.client.ServiceInvoker("eprmanager", "qwerty");
            invoker.deliverAsync(message);
		}
		catch (Exception ex)
		{
			fail();
		}

		RegistryUtil.unregister("eprmanager", "qwerty", epr);
		
		try
		{
            ServiceInvoker invoker = new org.jboss.soa.esb.client.ServiceInvoker("eprmanager", "qwerty");
            invoker.deliverAsync(message);
		}
		catch (Exception ex)
		{
		}
		
		File[] files = tmpDir.listFiles();
		
		for (int i = 0; i < files.length; i++)
		{
			if ((files[i].getName().endsWith(".unitProcessedOK")) ||
					(files[i].getName().equals("qwerty")))
			{
				try
				{
					files[i].delete();
				}
				catch (Exception ex)
				{
				}
			}
		}
	}

	private static String mDbDriver;

	private static String mDbUrl;

	private static String mDbUsername;

	private static String mDbPassword;

	private static Connection con;
}
