/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.message.tests;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;

import junit.framework.TestCase;

import org.jboss.internal.soa.esb.message.format.DeferredDeserialisationException;
import org.jboss.soa.esb.message.Attachment;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.Properties;
import org.jboss.soa.esb.util.Util;

/**
 * Unit tests for checking deserialisation of old formats and just in time resolving
 * of object values.
 * 
 * @author <a href='mailto:kevin.conner@jboss.com'>Kevin Conner</a>
 */

public class DeserializedValuesMessageUnitTest extends TestCase
{
    public void testOldSerializedAttachmentDeserialisation()
        throws Exception
    {
        final Attachment attachment = (Attachment)deserialise("old_attachment.ser") ;
        validateOldAttachment(attachment) ;
    }
    
    public void testOldSerializedBodyDeserialisation()
        throws Exception
    {
        final Body body = (Body)deserialise("old_body.ser") ;
        validateOldBody(body) ;
    }
    
    public void testOldSerializedPropertiesDeserialisation()
        throws Exception
    {
        final Properties properties = (Properties)deserialise("old_properties.ser") ;
        validateOldProperties(properties) ;
    }
    
    public void testOldSerializedMessageDeserialisation()
        throws Exception
    {
        final Message message = (Message)deserialise("old_message.ser") ;
        validateOldAttachment(message.getAttachment()) ;
        validateOldBody(message.getBody()) ;
        validateOldProperties(message.getProperties()) ;
    }
    
    public void testOldXMLMessageDeserialisation()
        throws Exception
    {
        final String contents = getContents("old_message.xml") ;
        final Message message = Util.deserialize(contents) ;
        validateOldAttachment(message.getAttachment()) ;
        validateOldBody(message.getBody()) ;
        validateOldProperties(message.getProperties()) ;
    }
    
    public void testNewSerializedAttachmentDeserialisation()
        throws Exception
    {
        final Attachment attachment = (Attachment)deserialise("new_attachment.ser") ;
        validateNewAttachment(attachment) ;
    }
    
    public void testNewSerializedBodyDeserialisation()
        throws Exception
    {
        final Body body = (Body)deserialise("new_body.ser") ;
        validateNewBody(body) ;
    }
    
    public void testNewSerializedPropertiesDeserialisation()
        throws Exception
    {
        final Properties properties = (Properties)deserialise("new_properties.ser") ;
        validateNewProperties(properties) ;
    }
    
    public void testNewSerializedMessageDeserialisation()
        throws Exception
    {
        final Message message = (Message)deserialise("new_message.ser") ;
        validateNewAttachment(message.getAttachment()) ;
        validateNewBody(message.getBody()) ;
        validateNewProperties(message.getProperties()) ;
    }
    
    public void testNewXMLMessageDeserialisation()
        throws Exception
    {
        final String contents = getContents("new_message.xml") ;
        final Message message = Util.deserialize(contents) ;
        validateNewAttachment(message.getAttachment()) ;
        validateNewBody(message.getBody()) ;
        validateNewProperties(message.getProperties()) ;
    }
    
    private void validateOldAttachment(final Attachment attachment)
    {
        assertEquals("unnamed count", 1, attachment.getUnnamedCount()) ;
        assertEquals("named count", 1, attachment.getNamedCount()) ;
        final Object namedAttachment = attachment.get("testAttribute") ;
        assertEquals("Named attachment", "Named value", namedAttachment) ;
        final Object unnamedAttachment = attachment.itemAt(0) ;
        assertEquals("Unnamed attachment", "Unnamed value", unnamedAttachment) ;
    }
    
    private void validateOldBody(final Body body)
    {
        final Object bodyValue = body.get() ;
        assertEquals("Body value", "Body value", bodyValue) ;
    }

    private void validateOldProperties(final Properties properties)
    {
        final Object propertyValue = properties.getProperty("testProperty") ;
        assertEquals("Property value", "Property value", propertyValue) ;
    }
    
    private void validateNewAttachment(final Attachment attachment)
    {
        assertEquals("unnamed count", 1, attachment.getUnnamedCount()) ;
        assertEquals("named count", 1, attachment.getNamedCount()) ;
        try
        {
            attachment.get("testAttribute") ;
            fail("Expected DeferredDeserialisationException for named attachment") ;
        }
        catch (final DeferredDeserialisationException dse) {}
        
        try
        {
            attachment.itemAt(0) ;
            fail("Expected DeferredDeserialisationException for unnamed attachment") ;
        }
        catch (final DeferredDeserialisationException dse) {}
    }
    
    private void validateNewBody(final Body body)
    {
        try
        {
            body.get() ;
            fail("Expected DeferredDeserialisationException for body contents") ;
        }
        catch (final DeferredDeserialisationException dse) {}
    }

    private void validateNewProperties(final Properties properties)
    {
        try
        {
            properties.getProperty("testProperty") ;
            fail("Expected DeferredDeserialisationException for named property") ;
        }
        catch (final DeferredDeserialisationException dse) {}
    }
    
    private Object deserialise(final String resource)
        throws IOException, ClassNotFoundException
    {
        final InputStream is = getClass().getResourceAsStream(resource) ;
        try
        {
            final ObjectInputStream ois = new ObjectInputStream(is) ;
            return ois.readObject() ;
        }
        finally
        {
            is.close() ;
        }
    }
    
    private String getContents(final String resource)
        throws IOException
    {
        final InputStream is = getClass().getResourceAsStream(resource) ;
        try
        {
            final InputStreamReader isr = new InputStreamReader(is) ;
            final StringBuilder sb = new StringBuilder() ;
            final char[] buffer = new char[256] ;
            while(true)
            {
                final int count = isr.read(buffer) ;
                if (count <= 0)
                {
                    break ;
                }
                sb.append(buffer, 0, count) ;
            }
            return sb.toString();
        }
        finally
        {
            is.close() ;
        }
    }
}
