/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.testutils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;

import junit.framework.TestCase;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPoolContainer;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.config.Generator;
import org.jboss.soa.esb.listeners.config.ConfigurationController;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleController;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.parameters.ParamRepositoryException;
import org.jboss.soa.esb.ConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.jms.JMSException;

/**
 * Utility class for working with the ESB configuration.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ESBConfigUtil {

    private Generator generator;
    private Document listenerConfig;
    private Document gatewayConfig;
    private ManagedLifecycleController controller;

    public ESBConfigUtil(InputStream esbXsdConfig) {
        AssertArgument.isNotNull(esbXsdConfig, "esbXsdConfig");
        ByteArrayOutputStream listenerConfigStream = new ByteArrayOutputStream();
        ByteArrayOutputStream gatewayConfigStream = new ByteArrayOutputStream();
        generator = null;

        try {
            generator = new Generator(esbXsdConfig, listenerConfigStream, gatewayConfigStream);
            generator.generate();
            listenerConfig = YADOMUtil.parseStream(new ByteArrayInputStream(listenerConfigStream.toByteArray()), false, false);
            gatewayConfig = YADOMUtil.parseStream(new ByteArrayInputStream(gatewayConfigStream.toByteArray()), false, false);
        } catch(Exception e) {
            e.printStackTrace();
            TestCase.fail(e.getMessage());
        }
    }

    public ConfigTree getListenerConfig(String listenerName) {
        AssertArgument.isNotNull(listenerName, "listenerName");
        return ConfigTree.fromElement(getListenerConfig(listenerName, listenerConfig));
    }

    public ConfigTree getGatewayConfig(String gatewayName) {
        AssertArgument.isNotNull(gatewayName, "gatewayName");
        return ConfigTree.fromElement(getListenerConfig(gatewayName, gatewayConfig));
    }

    private Element getListenerConfig(String name, Document config) {
        NodeList listenerConfigElements = config.getDocumentElement().getElementsByTagName(name);

        if(listenerConfigElements == null || listenerConfigElements.getLength() == 0) {
            return null;
        }

        return (Element) listenerConfigElements.item(0);
    }

    public ConfigTree getActionConfig(String listenerName, String actionName) {
        AssertArgument.isNotNull(listenerName, "listenerName");
        AssertArgument.isNotNull(actionName, "actionName");

        Element listener = getListenerConfig(listenerName, listenerConfig);
        NodeList actions = listener.getElementsByTagName("action");

        if(actions == null || actions.getLength() == 0) {
            return null;
        }

        for(int i = 0; i < actions.getLength(); i++) {
            Element action = (Element) actions.item(i);

            if(actionName.equals(action.getAttribute("action"))) {
                return ConfigTree.fromElement(action);
            }
        }
        
        return null;
    }

    public Document getListenerConfig() {
        return listenerConfig;
    }

    public Document getGatewayConfig() {
        return gatewayConfig;
    }

    public void startController() throws ParamRepositoryException, ConfigurationException, ManagedLifecycleException, SAXException {
        ConfigTree listenerConfigTree = ConfigTree.fromElement(listenerConfig.getDocumentElement());
        ConfigTree gatewayConfigTree = ConfigTree.fromElement(gatewayConfig.getDocumentElement());

        MockCourierFactory.install();
        MockRegistry.install();

        controller = ConfigurationController.startController(generator.getModel(), listenerConfigTree, gatewayConfigTree);
    }

    public void stopController() {
        try {
            ConfigurationController.stopController(controller);
        } finally {
            try {
                MockRegistry.uninstall();
            } finally {
                MockCourierFactory.uninstall();
            }
        }
    }
}
