/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.soap;

import org.milyn.delivery.dom.DOMElementVisitor;
import org.milyn.container.ExecutionContext;
import org.milyn.SmooksException;
import org.milyn.xml.DomUtils;
import org.milyn.cdr.SmooksResourceConfiguration;
import org.milyn.cdr.SmooksConfigurationException;
import org.w3c.dom.Element;
import org.apache.log4j.Logger;

import java.net.URI;
import java.net.URISyntaxException;

/**
 * Smooks transformer for transforming the WSDL for ESB endpoints.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class WsdlTransformer implements DOMElementVisitor {

    private static Logger logger = Logger.getLogger(WsdlTransformer.class);

    public static final String ENDPOINT_URI = "endpointURI";
    public static final String TARGET_CAT = "targetServiceCat";
    public static final String TARGET_NAME = "targetServiceName";
    public static final String TARGET_PROTOCOL = "targetProtocol";

    public void setConfiguration(SmooksResourceConfiguration smooksResourceConfiguration) throws SmooksConfigurationException {
    }

    public void visitBefore(Element element, ExecutionContext executionContext) throws SmooksException {
    }

    public void visitAfter(Element element, ExecutionContext executionContext) throws SmooksException {
        String elementName = DomUtils.getName(element);

        if(elementName.equals("import")) {
            transformImport(element, executionContext);
        } else if(elementName.equals("binding")) {
            transformBinding(element, executionContext);
        } else if(elementName.equals("address")) {
            transformAddress(element, executionContext);
        }
    }

    private void transformImport(Element element, ExecutionContext executionContext) {
        String location = DomUtils.getAttributeValue(element, "location");

        if(location != null) {
            try {
                URI locationURI = new URI(location);
                String targetServiceCat = (String) executionContext.getAttribute(WsdlTransformer.TARGET_CAT);
                String targetServiceName = (String) executionContext.getAttribute(WsdlTransformer.TARGET_NAME);
                String targetProtocol = (String) executionContext.getAttribute(WsdlTransformer.TARGET_PROTOCOL);
                String addParams = "serviceCat=" + targetServiceCat + "&serviceName=" + targetServiceName + "&protocol=" + targetProtocol;

                if(locationURI.getQuery() != null) {
                    element.setAttribute("location", (locationURI.toString() + "&" + addParams).replace("&", "&amp;"));
                } else {
                    element.setAttribute("location", (locationURI.toString() + "?" + addParams).replace("&", "&amp;"));
                }
            } catch (URISyntaxException e) {
                logger.info("Not processing WSDL import '" + location + "'.  Invalid URI.");
            }
        }
    }

    private void transformBinding(Element element, ExecutionContext executionContext) {
        URI endpointURI = (URI) executionContext.getAttribute(ENDPOINT_URI);
        if(endpointURI != null) {
            element.setAttribute("transport", "http://schemas.xmlsoap.org/soap/" + endpointURI.getScheme());
        }
    }

    private void transformAddress(Element element, ExecutionContext executionContext) {
        URI endpointURI = (URI) executionContext.getAttribute(ENDPOINT_URI);
        if(endpointURI != null) {
            // Make sure there's a path (if only a single slash).  JBossWS client code will
            // fail because of a feature of JBossRemoting.  See http://jira.jboss.com/jira/browse/JBWS-1802
            if(endpointURI.getPath() == null || endpointURI.getPath().equals("")) {
                try {
                    endpointURI = new URI(endpointURI.getScheme(), endpointURI.getUserInfo(),
                            endpointURI.getHost(), endpointURI.getPort(), "/", endpointURI.getQuery(), endpointURI.getFragment());
                } catch (URISyntaxException e) {
                    logger.error("Failed to properly format endpoint location URL '" + endpointURI.toString() + "'.  Failed to add single slash character for path.  This may cause JBossWS Client code to fail.");
                }
            }
            
            element.setAttribute("location", endpointURI.toString());
        }
    }
}
