/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

/**
 * Abstract class for Actions that make calls to Spring beans. 
 * 
 * <p>This class creates a Spring IoC from a jboss-esb.xml element named "springContextXml". 
 * A comma separated list of spring context files may be used.
 * 
 * @author <a href="mailto:james.williams@redhat.com">James Williams</a>.
 * 
 */
public abstract class AbstractSpringAction extends AbstractActionLifecycle
      implements ActionLifecycle
{
   private static Logger logger = Logger.getLogger(AbstractSpringAction.class);

   private static final String SPRING_CONTEXT_XML_ATTR = "springContextXml";

   private String springContextXml;

   protected ConfigTree configTree;

   private BeanFactory factory;

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.soa.esb.actions.AbstractActionLifecycle#initialise()
    */
   public void initialise() throws ActionLifecycleException
   {
      springContextXml = configTree.getAttribute(SPRING_CONTEXT_XML_ATTR);

      if (springContextXml == null)
      {
         logger.error("No Spring context specified on action config "
               + configTree.getAttribute("name")
               + ". Expecting Spring context name.");
         throw new ActionLifecycleException(
               "No Spring context specified on action config: "
                     + SPRING_CONTEXT_XML_ATTR + ".");
      } else
      {
         initializeSpring();
      }
   }

   /**
    * Initialize Spring IoC
    * 
    * @throws ActionLifecycleException
    */
   protected void initializeSpring() throws ActionLifecycleException
   {
      if (isBeanFactoryNull())
      {
         loadSpringIoc();
      }
   }

   /**
    * Request a BeanFactory instance from the action. If no Spring IoC container
    * exists, create one.
    * 
    * @return Spring Bean Factory
    * @throws ActionLifecycleException
    */
   protected BeanFactory getBeanFactory() throws ActionLifecycleException
   {
      return this.factory;
   }

   /**
    * Check to see if Spring Bean factory is null. Mostly used for unit tests,
    * but could provide use in other situations.
    * 
    * @return
    */
   public boolean isBeanFactoryNull()
   {
      if (factory == null)
      {
         return true;
      } else
      {
         return false;
      }
   }

   /**
    * Create a spring IoC container.
    * 
    * @throws ActionLifecycleException
    */
   private void loadSpringIoc() throws ActionLifecycleException
   {
      String springContextXml = configTree
            .getAttribute(SPRING_CONTEXT_XML_ATTR);
      try
      {
         ApplicationContext springContext;

         if (springContextXml.contains(","))
         {
            String[] springContextXmls = springContextXml.split(",");
            springContext = new ClassPathXmlApplicationContext(
                  springContextXmls);
         } else
         {
            springContext = new ClassPathXmlApplicationContext(springContextXml);
         }

         factory = (BeanFactory) springContext;
      } catch (Exception e)
      {
         throw new ActionLifecycleException(e);
      }
   }

   /**
    * Generic Exception handler for Spring Actions. Displays the root cause
    * message and full stack trace.
    * 
    * @param message
    * @param exception
    */
   public void exceptionHandler(Message message, Throwable exception)
   {
      Throwable rootCause = exception.getCause();
      StackTraceElement[] traceElms = rootCause.getStackTrace();

      StringBuffer stackTrace = new StringBuffer("Exception Root Cause is: \n");
      stackTrace.append(rootCause.getMessage());
      stackTrace.append("\n Full Stack Trace is: \n");
      for (StackTraceElement elm : traceElms)
      {
         stackTrace.append(elm);
         stackTrace.append("\n");
      }

      logger.error(stackTrace.toString());
   }

   /**
    * Conviencence method for printing a console log header
    */
   protected void logHeader()
   {
      System.out.println("\n&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
   }

   /**
    * Conviencence method for printing a console log footer
    */
   protected void logFooter()
   {
      System.out.println("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&\n");
   }

   /* (non-Javadoc)
    * @see org.jboss.soa.esb.actions.AbstractActionLifecycle#destroy()
    */
   public void destroy() throws ActionLifecycleException
   {
   }
}
