/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.monitoring.server;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.DeliverOnlyCourier;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierFactory;
import org.jboss.soa.esb.couriers.CourierUtil;
import org.jboss.soa.esb.listeners.RegistryUtil;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.body.content.MessageBodyTypeNames;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;
import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;

/**
 * DataFilerJob is a Quartz job which, when fired, delivers a message to
 * all DataCollector services asking for statistics.
 * 
 * @author <a href="mailto:tcunning@redhat.com">Tom Cunningham</a>
 * @since Version 4.2
 */
public class DataFilerJob implements Job {
	private static Logger logger = Logger.getLogger(DataFilerJob.class);
	
	public static final String JOB_GROUP = "ESB";
	public static final String JOB_NAME = "DataFiler";
	
	public void execute (JobExecutionContext jobExecutionContext) 
		throws JobExecutionException {
	        final ClassLoader currentClassLoader = Thread.currentThread().getContextClassLoader() ;
	        try
	        {
        	        Thread.currentThread().setContextClassLoader((ClassLoader)jobExecutionContext.getJobDetail().getJobDataMap().get(ClassLoader.class.getName())) ;
        		DeliverOnlyCourier sender = null;
        		logger.debug("Collect statistics from EPRs");
        		List list;
        		try {
        			list = RegistryUtil.getEprs(FilerAction.INTERNAL_CATEGORY, FilerAction.DATA_COLLECTOR_SERVICE_ACTION_NAME);
        			if (list != null) {
        				for (int i = 0; i < list.size(); i++) {
        					EPR epr = (EPR)list.get(i); 
        					// Send message requesting performance data
        					Message esbMessage = MessageFactory.getInstance().getMessage();
        					esbMessage.getProperties().setProperty(MessageBodyTypeNames.COMMAND_TYPE, MessageBodyTypeNames.STATISTICS_REQUEST); 
        					esbMessage.getBody().add(BytesBody.BYTES_LOCATION, "get statistics".getBytes());
        					
        				    URI uri = new URI(UUID.randomUUID().toString());
        				    esbMessage.getHeader().getCall().setMessageID(uri);
        
        				    try {
        						sender = CourierFactory.getCourier(epr);
        				    	sender.deliver(esbMessage);				    				    	
        				    } finally {
        				    	CourierUtil.cleanCourier(sender);
        				    }
        				}
        			}
        			
        			list = RegistryUtil.getEprs(FilerAction.INTERNAL_CATEGORY, FilerAction.OPERATIONS_COLLECTOR_SERVICE_ACTION_NAME);			
        			if (list != null) {
        				for (int i = 0; i < list.size(); i++) {
        					EPR epr = (EPR)list.get(i); 
        					// Send message requesting operations data
        					Message opMessage = MessageFactory.getInstance().getMessage();
        					opMessage.getProperties().setProperty(MessageBodyTypeNames.COMMAND_TYPE, MessageBodyTypeNames.OPERATIONS_REQUEST); 
        					opMessage.getBody().add(BytesBody.BYTES_LOCATION, "get statistics".getBytes());
        				
        					URI opURI = new URI(UUID.randomUUID().toString());
        					opMessage.getHeader().getCall().setMessageID(opURI);
        					try{
        						sender = CourierFactory.getCourier(epr);
        						sender.deliver(opMessage);
        					} finally {
        						CourierUtil.cleanCourier(sender);
        					}
        				}
        			}
        		} catch (RegistryException e) {
        			logger.error("", e);
        		} catch (CourierException e) {
        			logger.error("", e);
        		} catch (MalformedEPRException e) {
        			logger.error("", e);
        		} catch (URISyntaxException e) {
        			logger.error("", e);
        		} catch (ServiceNotFoundException e) {
        			logger.error("", e);
        		}
	        }
	        finally
	        {
                    Thread.currentThread().setContextClassLoader(currentClassLoader) ;
	        }
	}
}
