/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.admin.console.transform.importexport;

import static org.jboss.seam.ScopeType.CONVERSATION;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.net.URISyntaxException;

import javax.persistence.EntityManager;

import org.apache.log4j.Logger;
import org.jboss.seam.annotations.Begin;
import org.jboss.seam.annotations.End;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Out;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.core.FacesMessages;
import org.jboss.soa.esb.admin.console.UpdateNotifier;
import org.jboss.soa.esb.admin.console.transform.NewResource;
import org.jboss.soa.esb.admin.console.transform.SmooksResourceList;
import org.jboss.soa.esb.admin.console.transform.SmooksUtils;
import org.jboss.soa.esb.admin.console.transform.TransformationResource;
import org.jboss.soa.esb.admin.console.transform.TransformationResourceParameter;
import org.milyn.cdr.Parameter;
import org.milyn.cdr.ProfileTargetingExpression;
import org.milyn.cdr.SmooksResourceConfiguration;
import org.milyn.cdr.SmooksResourceConfigurationList;
import org.milyn.cdr.XMLConfigDigester;
import org.xml.sax.SAXException;

/**
 * Imports/Exports Smooks transformation resource condifurations.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
@Name("resourceImportExport")
@Scope(CONVERSATION)
public class ResourceImportExport implements Serializable {

	private static final long serialVersionUID = 1L;

	private static Logger logger = Logger.getLogger(ResourceImportExport.class);
	
	@In(create=true)
	private EntityManager entityManager;

	@In(create=true)
	private UpdateNotifier updateNotifier;
	
	private byte[] importData;
	
	@Out(required=false)
	private String exportData;

	@Begin(join=true)
	public String show() {
		importData = null;
		exportData = null;
		
		return "import-export-resources";
	}
	
	public void importResources() {	
		if(importData != null) {
			try {
				loadConfigurations(new ByteArrayInputStream(importData));
			} catch (SAXException e) {
				FacesMessages.instance().add("Import Failed.  Invalid Configuration: " + e.getMessage());
				logger.warn("Import Failed.  Invalid Configuration: " + e.getMessage());
				return;
			} catch (IOException e) {
				FacesMessages.instance().add("Import Failed.  Error Reading Configuration: " + e.getMessage());
				logger.warn("Import Failed.  Error Reading Configuration: " + e.getMessage());
				return;
			}
			
			logger.info("Successfully imported resources.");
			if(logger.isDebugEnabled()) {
				logger.debug("Imported:\n[" + new String(importData) + "]");
			}
			updateNotifier.sendNotification();
		} else {
			logger.warn("Resource imported request received, but no import data!!");
		}
	}
	
	public void exportResources() {
		exportData = SmooksResourceList.getList(entityManager);
		logger.info("Successfully exported resources.");
	}
	
	@End
	public String home() {
		return "home";
	}

	private void loadConfigurations(InputStream resStream) throws SAXException, IOException {
        SmooksResourceConfigurationList resList = null;
        try {
            resList = XMLConfigDigester.digestConfig(resStream, "import-list");
            for(int i = 0; i < resList.size(); i++) {
                TransformationResource transResource = toTransformationResource(resList.get(i));

                NewResource.persistNewResource(transResource, entityManager);
            }
        } catch (URISyntaxException e) {
            throw new IOException("Unexpected URISyntaxException: " + e.getMessage());
        }
	}

	private TransformationResource toTransformationResource(SmooksResourceConfiguration configuration) {
		TransformationResource resource = new TransformationResource();
		
		resource.setUseragent(getUseragentString(configuration.getProfileTargetingExpressions()));
		resource.setSelector(configuration.getSelector());
		resource.setPath(configuration.getResource());
		resource.setNamespace(configuration.getSelectorNamespaceURI());
		
		List smooksParams = configuration.getParameters();
		resource.setParameters(new ArrayList<TransformationResourceParameter>());
		for(int i = 0; i < smooksParams.size(); i++) {
			Object listEntry = smooksParams.get(i);
			if(listEntry instanceof List) {
				List parameterGroup = (List)listEntry; 
				for(int ii = 0; ii < parameterGroup.size(); ii++) {
					Parameter param = (Parameter)parameterGroup.get(ii);
					addParameter(resource, param);
				}
			} else if(listEntry instanceof Parameter) {
				Parameter param = (Parameter)listEntry;
				addParameter(resource, param);
			}
		}
		
		return resource;
	}

	private void addParameter(TransformationResource resource, Parameter param) {
		if(param.getName().equals(SmooksUtils.RESOURCE_NAME_PARAM)) {
			// This is not a "real" parameter.  It's just how we get over the fact that the SmooksResourceConfiguration
			// class doesn't have "name" attribute like the TransformationResource class.  So, we import and export
			// as a resource parameter.
			resource.setName(param.getValue());
		} else {
			TransformationResourceParameter parameter;
			parameter = new TransformationResourceParameter(param.getName(), param.getType(), param.getValue());
			parameter.setTransResource(resource);
			resource.getParameters().add(parameter);
		}
	}

	
	private String getUseragentString(ProfileTargetingExpression[] profileTargetingExpressions) {
		StringBuffer string = new StringBuffer();
		
		for(int i = 0; i < profileTargetingExpressions.length; i++) {
			string.append(profileTargetingExpressions[i].getExpression());
			if(i < (profileTargetingExpressions.length - 1)) {
				string.append(", ");
			}
		}
		
		return string.toString();
	}

	public byte[] getImportData() {
		return importData;
	}

	public void setImportData(byte[] importData) {
		this.importData = importData;
	}

	public String getExportData() {
		return exportData;
	}

	public void setExportData(String exportData) {
		this.exportData = exportData;
	}
}
