/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2007
 */

package org.jboss.soa.esb.eclipse.wizards;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileInfo;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.jboss.soa.esb.eclipse.util.ESBUtils;

/**
 * The wizard page for configuring  the ESB install and application server locations.
 * 
 * @author kevin
 */
public class ESBServerPage extends WizardPage
{
    /**
     * The default text width.
     */
    private static final int TEXT_WIDTH = 250 ;
    /**
     * The text for the browse button.
     */
    private static final String BROWSE_TEXT = "Browse" ;
    
    /**
     * The initial value of the ESB location.
     */
    private String initialESBLocation ;
    /**
     * The initial value of the AS location.
     */
    private String initialASLocation ;
    
    /**
     * The initial value of the AS Profile location.
     */
    private String initialASProfileLocation ;
    
    /**
     * The ESB location label.
     */
    private Label esbLocationLabel ;
    /**
     * The AS location label.
     */
    private Label asLocationLabel ;
    
    /**
     * The AS Configlocation label.
     */
    private Label asProfileLocationLabel ;
    
    /**
     * The ESB location text field.
     */
    private Text esbLocationField ;
    /**
     * The AS location text field.
     */
    private Text asLocationField ;
    /**
     * The AS Profile location combo
     */
    private Combo asProfileLocationCombo;
    
    /**
     * The ESB browse button.
     */
    private Button esbLocationButton ;
    /**
     * The AS browse button.
     */
    private Button asLocationButton ;
    /**
     * Is the ESB location valid?
     */
    private boolean esbLocationValid = false;
    /**
     * Is the AS location enabled?
     */
    private boolean asEnabled = true ;
    /**
     * Is the AS location valid?
     */
    private boolean asLocationValid = false;
    /**
     * Is the profile selected?
     */
    private boolean asProfileSelected = false;
    /**
     * Construct the wizard page.
     * @param pageName The page name.
     */
    public ESBServerPage(final String pageName)
    {
        super(pageName) ;
    }
    
    /**
     * Create the control for the page.
     * @param parent The parent composite.
     */
    public void createControl(final Composite parent)
    {
        final Composite pageComposite = new Composite(parent, SWT.NULL) ;
        pageComposite.setFont(parent.getFont()) ;
        initializeDialogUnits(parent) ;
        
        pageComposite.setLayout(new GridLayout()) ;
        pageComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true)) ;
        
        final Composite dataComposite = new Composite(pageComposite, SWT.NONE) ;
        dataComposite.setLayout(new GridLayout(3, false)) ;
        dataComposite.setLayoutData(new GridData(SWT.FILL, GridData.CENTER, true, false)) ;
        
        addEsbLocation(parent, dataComposite);
        
        addAsLocation(parent, dataComposite);
        
        addProfileLocation(parent, dataComposite);
        
        setErrorMessage(null);
        setMessage(null);
        
        if (initialESBLocation != null)
        {
            esbLocationField.setText(initialESBLocation) ;
        }
        if (asEnabled && (initialASLocation != null))
        {
        	asLocationField.setText(initialASLocation) ;
        	// validation occured in the pref pane
        	asLocationValid = true;
        	asLocationField.setEnabled(asLocationValid);
        }
        if (initialASProfileLocation != null){
        	asProfileLocationCombo.setText(initialASProfileLocation);
        	// validation occured in the pref pane
        	asProfileSelected = true;
        	asProfileLocationLabel.setVisible(asProfileSelected);
        	asProfileLocationCombo.setVisible(asProfileSelected);
        	asProfileLocationCombo.setEnabled(asProfileSelected);
        }
        dataComposite.pack(true);
        pageComposite.pack(true);
        setControl(pageComposite); 
    }
    
    /**
     * Add the AS Profile widgets
     * @param parent
     * @param dataComposite
     */
	private void addProfileLocation(final Composite parent, final Composite dataComposite) {
		asProfileLocationLabel = new Label(dataComposite, SWT.NONE) ;
		asProfileLocationLabel.setVisible(asLocationValid);
		asProfileLocationLabel.setFont(parent.getFont()) ;
        asProfileLocationLabel.setText("Application Server Profile Location:") ;
        
        asProfileLocationCombo = new Combo(dataComposite, SWT.READ_ONLY);
        asProfileLocationCombo.setVisible(asLocationValid);
        asProfileLocationCombo.setFont(parent.getFont());
        asProfileLocationCombo.pack();
        asProfileLocationCombo.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(final SelectionEvent event) {
        		asProfileSelected() ;
            }
        	
        });
	}
	
	/**
	 * Add the Application Server widgets
	 * @param parent
	 * @param dataComposite
	 */
	private void addAsLocation(final Composite parent, final Composite dataComposite) {
		asLocationLabel = new Label(dataComposite, SWT.NONE) ;
        asLocationLabel.setFont(parent.getFont()) ;
        asLocationLabel.setText("Application Server Location:") ;
        
        asLocationField = new Text(dataComposite, SWT.BORDER) ;
        asLocationField.setLayoutData(new GridData(TEXT_WIDTH, SWT.DEFAULT)) ;
        asLocationField.setFont(parent.getFont()) ;
        asLocationField.setEnabled(asEnabled);
        asLocationField.setText("Please choose esb location first");
        asLocationField.addModifyListener(new ModifyListener() {
            public void modifyText(final ModifyEvent event) {
                validateASLocation() ;
            }
        }) ;
        
        asLocationButton = new Button(dataComposite, SWT.PUSH) ;
        asLocationButton.setText(BROWSE_TEXT) ;
        asLocationButton.setEnabled(asEnabled);
        asLocationButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                handleBrowse(asLocationField, "Application Server Location") ;
            }
        }) ;
	}

	/**
	 * Add the ESB Server widgets
	 * @param parent
	 * @param dataComposite
	 */
	private void addEsbLocation(final Composite parent, final Composite dataComposite) {
		esbLocationLabel = new Label(dataComposite, SWT.NONE) ;
        esbLocationLabel.setFont(parent.getFont()) ;
        esbLocationLabel.setText("ESB Install Location:") ;
        
        esbLocationField = new Text(dataComposite, SWT.BORDER) ;
        esbLocationField.setLayoutData(new GridData(TEXT_WIDTH, SWT.DEFAULT)) ;
        esbLocationField.setFont(parent.getFont()) ;
        esbLocationField.addModifyListener(new ModifyListener() {
            public void modifyText(final ModifyEvent event) {
                validateESBLocation() ;
            }
        }) ;
        
        esbLocationButton = new Button(dataComposite, SWT.PUSH) ;
        esbLocationButton.setText(BROWSE_TEXT) ;
        esbLocationButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                handleBrowse(esbLocationField, "ESB Install Location") ;
            }
        }) ;
	}
    
    /**
     * Get the AS Location
	 * populate from folders under the servers directory 
     **/
    public void populateProfileSelection() {
		asProfileLocationCombo.setItems(ESBUtils.getValidProfiles(asLocationField.getText()));
		asProfileLocationCombo.pack();
	}

    /**
     * Set the initial value of the ESB location.
     * @param initialESBLocation The initial value.
     */
    public void setInitialESBLocation(final String initialESBLocation)
    {
        this.initialESBLocation = initialESBLocation ;
    }
    
    /**
     * Set the initial value of the AS location.
     * @param initialASLocation The initial value.
     */
    public void setInitialASLocation(final String initialASLocation)
    {
        this.initialASLocation = initialASLocation ;
    }
    
    /**
     * Validate the ESB location.
     */
    private void validateESBLocation()
    {
        final String esbLocation = esbLocationField.getText() ;
        
        esbLocationValid = ESBUtils.isESBServerOrESBLocation(esbLocation) ;
        if (!esbLocationValid)
        {
            setErrorMessage("Invalid ESB Install Location") ;
        }
        else
        {
            setErrorMessage(null) ;
        }
        
        final boolean enabled = !(esbLocationValid && ESBUtils.isESBServerLocation(esbLocation)) ;
        if (enabled != asEnabled)
        {
            asEnabled = enabled ;
            asLocationField.setText("") ;
            asLocationLabel.setEnabled(enabled) ;
            asLocationField.setEnabled(enabled) ;
            asLocationButton.setEnabled(enabled) ;
        }
        
        if (esbLocationValid && asEnabled && !asLocationValid)
        {
            setErrorMessage("Invalid Application Server Install Location") ;
        }
        setPageComplete(esbLocationValid && (!asEnabled || asLocationValid)) ;
    }
    
    /**
     * Validate the AS location.
     */
    private void validateASLocation()
    {
        if (asEnabled)
        {
            final String asLocation = asLocationField.getText() ;
            
            asLocationValid = ESBUtils.isASLocation(asLocation) ;
            
            if (esbLocationValid)
            {
            	populateProfileSelection();
                
                if (!asLocationValid)
                {
                    setErrorMessage("Invalid Application Server Install Location") ;
                }
                else
                {
                	asProfileLocationCombo.setVisible(asLocationValid);
                	asProfileLocationLabel.setVisible(asLocationValid);
                    setErrorMessage(null) ;
                }
            }
        }
        
    }
    
    /**
     * Indicate the AS Profile location has been selected.
     */
    private void asProfileSelected()
    {
    	asProfileSelected = true;
        if (asEnabled)
        {
        	
            setPageComplete(asProfileSelected) ;
            setErrorMessage(null) ;
        }
        
    }    
    
    /**
     * Handle the browse button for the specified location field.
     * @param locationField The location field.
     */
    private void handleBrowse(final Text locationField, final String message)
    {
        final String path = getPath(locationField) ;
        final DirectoryDialog directoryDialog = new DirectoryDialog(locationField.getShell()) ;
        directoryDialog.setMessage(message) ;
        directoryDialog.setFilterPath(path) ;
        final String directory = directoryDialog.open() ;
        if (directory != null)
        {
            locationField.setText(directory) ;
        }
    }
    
    /**
     * Get the path value for the directory dialog.
     * @param locationField The text field containing the path.
     * @return The path value if the directory exists, empty string otherwise.
     */
    private String getPath(final Text locationField)
    {
        final String pathValue = locationField.getText() ;
        if (pathValue.length() > 0)
        {
            final IPath path = Path.fromOSString(pathValue) ;
            final IFileStore pathFileStore = EFS.getLocalFileSystem().getStore(path) ;
            final IFileInfo pathFileInfo = pathFileStore.fetchInfo() ;
            
            if (pathFileInfo.exists())
            {
                return pathValue ;
            }
        }
        return "" ;
    }
    
    /**
     * Get the value of the ESB location.
     * @return The ESB location or null if not valid.
     */
    public String getESBLocation()
    {
        if ((esbLocationField != null) && esbLocationValid)
        {
            return esbLocationField.getText() ;
        }
        return null ;
    }
    
    /**
     * Get the value of the AS location.
     * @return The AS location or null if not valid.
     */
    public String getASLocation()
    {
        if ((asLocationField != null) && asLocationValid)
        {
            return asLocationField.getText() ;
        }
        return null ;
    }
    
    
    /**
     * Get the value of the AS Profile location.
     * @return The AS Profile location or null if not valid.
     */
    public String getASProfileLocation()
    {
        if (asProfileLocationCombo != null)
        {
            return asProfileLocationCombo.getText() ;
        }
        return null ;
    }

	public String getInitialASProfileLocation() {
		return initialASProfileLocation;
	}

	public void setInitialASProfileLocation(String initialASProfileLocation) {
		this.initialASProfileLocation = initialASProfileLocation;
	}
}
