/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.app;

import java.net.URL;

import org.xml.sax.InputSource;

import org.jbpm.JbpmContext;
import org.jbpm.bpel.BpelException;
import org.jbpm.bpel.graph.def.BpelProcessDefinition;
import org.jbpm.bpel.integration.catalog.ServiceCatalog;
import org.jbpm.bpel.persistence.db.BpelGraphSession;
import org.jbpm.bpel.xml.AppDescriptorReader;
import org.jbpm.bpel.xml.ProblemCollector;
import org.jbpm.configuration.ObjectFactory;
import org.jbpm.jpdl.xml.Problem;

/**
 * Binding of <tt>bpelApplication</tt> element.
 * @author Alejandro Guzar
 * @version $Revision: 1.8 $ $Date: 2007/08/21 09:33:19 $
 */
public class AppDescriptor extends AppScope {

  private String targetNamespace;
  private Integer version;
  private ServiceCatalog serviceCatalog;

  public static final String RESOURCE_APP_DESCRIPTOR = "resource.app.descriptor";
  public static final String DEFAULT_APP_DESCRIPTOR = "/bpel-application.xml";

  public String getTargetNamespace() {
    return targetNamespace;
  }

  public void setTargetNamespace(String targetNamespace) {
    this.targetNamespace = targetNamespace;
  }

  public Integer getVersion() {
    return version;
  }

  public void setVersion(Integer version) {
    this.version = version;
  }

  public ServiceCatalog getServiceCatalog() {
    return serviceCatalog;
  }

  public void setServiceCatalog(ServiceCatalog serviceCatalog) {
    this.serviceCatalog = serviceCatalog;
  }

  public BpelProcessDefinition findProcessDefinition(JbpmContext jbpmContext) {
    // this descriptor references a particular process definition
    String name = getName();
    String targetNamespace = getTargetNamespace();
    Integer version = getVersion();

    // use the given context to find the definition
    BpelGraphSession graphSession = BpelGraphSession.getInstance(jbpmContext);
    BpelProcessDefinition processDefinition;
    // check for a version indicator
    if (version != null) {
      // find a specific version of the process
      processDefinition = graphSession.findProcessDefinition(name, targetNamespace,
          version.intValue());
    }
    else {
      // just retrieve the latest version
      processDefinition = graphSession.findLatestProcessDefinition(name, targetNamespace);
    }

    // if no such process exist, halt
    if (processDefinition == null)
      throw new BpelException("process not found: name=" + name + ", version=" + version);

    return processDefinition;
  }

  public void accept(AppDescriptorVisitor visitor) {
    visitor.visit(this);
  }

  public static AppDescriptor readAppDescriptor(JbpmContext jbpmContext) {
    String appDescriptorResource = DEFAULT_APP_DESCRIPTOR;

    ObjectFactory objectFactory = jbpmContext.getObjectFactory();
    if (objectFactory.hasObject(RESOURCE_APP_DESCRIPTOR)) {
      // use the configured resource instead
      appDescriptorResource = (String) objectFactory.createObject(AppDescriptor.RESOURCE_APP_DESCRIPTOR);
    }

    // locate the app descriptor using the context class loader
    URL appDescriptorURL = Thread.currentThread().getContextClassLoader().getResource(
        appDescriptorResource);

    // could the app descriptor be found in the context resources?
    if (appDescriptorURL == null) {
      // fall back to the loader of this class
      appDescriptorURL = AppDescriptor.class.getResource(appDescriptorResource);

      // if the descriptor is really not there, halt
      if (appDescriptorURL == null)
        throw new BpelException("could not find application descriptor: " + appDescriptorResource);
    }

    // get the shared reader
    AppDescriptorReader appDescriptorReader = new AppDescriptorReader();

    // prepare custom error handling
    ProblemCollector problemHandler = new ProblemCollector(appDescriptorResource);
    appDescriptorReader.setProblemHandler(problemHandler);

    // parse content
    AppDescriptor appDescriptor = new AppDescriptor();
    appDescriptorReader.read(appDescriptor, new InputSource(appDescriptorURL.toExternalForm()));

    // if the descriptor has errors, halt
    if (Problem.containsProblemsOfLevel(problemHandler.getProblems(), Problem.LEVEL_ERROR)) {
      throw new BpelException("errors found in bpel application descriptor");
    }

    return appDescriptor;
  }
}