/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.graph.def;

import java.io.Serializable;

import org.apache.commons.lang.builder.ToStringBuilder;

import org.jbpm.bpel.graph.exe.BpelFaultException;
import org.jbpm.bpel.graph.exe.LinkInstance;
import org.jbpm.bpel.graph.struct.Flow;
import org.jbpm.bpel.sublang.def.Expression;
import org.jbpm.bpel.xml.util.DatatypeUtil;
import org.jbpm.context.exe.ContextInstance;
import org.jbpm.graph.exe.Token;

/**
 * Establishes explicit control dependencies between nested child activities
 * within a {@link Flow flow}.
 * @author Juan Cant
 * @version $Revision: 1.3 $ $Date: 2007/02/19 22:25:45 $
 */
public class LinkDefinition implements Serializable {

  long id;
  private String name;
  private Activity target;
  private Activity source;
  private Expression transitionCondition;

  private static final long serialVersionUID = 1L;

  public LinkDefinition() {
  }

  public LinkDefinition(String name) {
    setName(name);
  }

  public void determineStatus(Token token) {
    LinkInstance linkInstance = getInstance(token);

    if (transitionCondition == null) {
      linkInstance.statusDetermined(true);
    }
    else {
      try {
        // uses the source variable context to evaluate the transition condition
        linkInstance.statusDetermined(DatatypeUtil.toBoolean(transitionCondition.getEvaluator()
            .evaluate(token)));
      }
      catch (BpelFaultException e) {
        // set a negative link status before throwing the exception
        linkInstance.setStatus(Boolean.FALSE);
        throw e;
      }
    }
  }

  public Expression getTransitionCondition() {
    return transitionCondition;
  }

  public void setTransitionCondition(Expression transitionCondition) {
    this.transitionCondition = transitionCondition;
  }

  public String getName() {
    return name;
  }

  public void setName(String name) {
    this.name = name;
  }

  public Activity getTarget() {
    return target;
  }

  void setTarget(Activity target) {
    this.target = target;
  }

  public Activity getSource() {
    return source;
  }

  void setSource(Activity source) {
    this.source = source;
  }

  public String toString() {
    return new ToStringBuilder(this).append("name", name).append("source",
        source).append("target", target).toString();
  }

  public LinkInstance getInstance(Token token) {
    ContextInstance ci = token.getProcessInstance().getContextInstance();
    return (LinkInstance) ci.getVariable(name, token);
  }

  public LinkInstance createInstance(Token token) {
    ContextInstance ci = token.getProcessInstance().getContextInstance();
    LinkInstance linkInstance = new LinkInstance(this);
    ci.createVariable(name, linkInstance, token);
    return linkInstance;
  }
}