package org.jbpm.bpel.tools;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import org.jbpm.bpel.graph.def.BpelProcessDefinition;
import org.jbpm.bpel.xml.ProblemCounter;
import org.jbpm.bpel.xml.ProblemHandler;
import org.jbpm.file.def.FileDefinition;
import org.jbpm.jpdl.xml.Problem;

/**
 * @author Alejandro Guizar
 * @version $Revision: 1.2 $ $Date: 2007/09/10 23:44:45 $
 */
public class WebModuleBuilder {

  private File outputDirectory = FileUtil.TEMPORARY_DIRECTORY;
  private String moduleName;
  private String packagePrefix = "org.example";
  private ProblemHandler problemHandler = new ProblemCounter();

  static final String DIR_WSDL = "wsdl";
  static final String DIR_WEB_INF = "WEB-INF";
  static final String DIR_CLASSES = "classes";
  static final char SEPARATOR = '/'; // use File.separatorChar instead?

  public File getOutputDirectory() {
    return outputDirectory;
  }

  public void setOutputDirectory(File outputDirectory) {
    this.outputDirectory = outputDirectory;
  }

  public String getModuleName() {
    return moduleName;
  }

  public void setModuleName(String moduleName) {
    this.moduleName = moduleName;
  }

  public ProblemHandler getProblemHandler() {
    return problemHandler;
  }

  public void setProblemHandler(ProblemHandler problemHandler) {
    if (problemHandler == null)
      throw new IllegalArgumentException("problem handler cannot be null");

    this.problemHandler = problemHandler;
  }

  public void generateWebModule(BpelProcessDefinition processDefinition) {
    if (moduleName == null)
      moduleName = toLowerCaseJavaIdentifier(processDefinition.getName());

    callWsdlServiceGenerator(processDefinition);
    if (problemHandler.getProblemCount() > 0)
      return;

    callJavaMappingGenerator();
    if (problemHandler.getProblemCount() > 0)
      return;

    try {
      assembleWebModule(processDefinition);
    }
    catch (IOException e) {
      Problem problem = new Problem(Problem.LEVEL_ERROR, "web module assembly failed", e);
      problemHandler.add(problem);
    }
  }

  protected File getWebDirectory() {
    return new File(outputDirectory, "web");
  }

  protected File getClassesDirectory() {
    File classesDirectory = new File(outputDirectory, DIR_CLASSES);
    classesDirectory.mkdir();
    return classesDirectory;
  }

  protected File getWsdlDirectory() {
    return new File(getWebDirectory(), DIR_WSDL);
  }

  protected String getWsdlBindingFilesPrefix() {
    return moduleName + "-binding-";
  }

  protected String getWsdlServiceFileName() {
    return moduleName + "-service.wsdl";
  }

  protected File getJaxRpcMappingFile() {
    return new File(getWebDirectory(), "jaxrpc-mapping.xml");
  }

  protected File getWebModuleFile() {
    return new File(outputDirectory, moduleName + ".war");
  }

  protected void callWsdlServiceGenerator(BpelProcessDefinition processDefinition) {
    WsdlServiceTool tool = new WsdlServiceTool();
    tool.setWsdlDirectory(getWsdlDirectory());
    tool.setBindingFilesPrefix(getWsdlBindingFilesPrefix());
    tool.setServiceFileName(getWsdlServiceFileName());
    tool.setProblemHandler(problemHandler);

    tool.generateWsdlService(processDefinition);
  }

  protected void callJavaMappingGenerator() {
    JavaMappingTool tool = new WscompileTool();
    tool.setWsdlFile(new File(getWsdlDirectory(), getWsdlServiceFileName()));
    tool.setPackageName(generateJavaMappingPackage());
    tool.setJaxrpcMappingFile(getJaxRpcMappingFile());
    tool.setClassesDirectory(getClassesDirectory());
    tool.setProblemHandler(problemHandler);

    tool.generateJavaMapping();
  }

  protected String generateJavaMappingPackage() {
    return packagePrefix + '.' + moduleName;
  }

  private static String toLowerCaseJavaIdentifier(String name) {
    int i;
    final int n = name.length();
    char[] identifierBuffer = new char[n];

    // first character
    for (i = 0; i < n; i++) {
      char ch = name.charAt(i);
      if (Character.isJavaIdentifierStart(ch)) {
        identifierBuffer[0] = Character.toLowerCase(ch);
        break;
      }
    }

    // next characters
    int k = 1;
    for (i++; i < n; i++) {
      char ch = name.charAt(i);
      if (Character.isJavaIdentifierPart(ch))
        identifierBuffer[k++] = Character.toLowerCase(ch);
    }

    return new String(identifierBuffer, 0, k);
  }

  protected void assembleWebModule(BpelProcessDefinition processDefinition) throws IOException {
    ZipOutputStream moduleOut = new ZipOutputStream(new FileOutputStream(getWebModuleFile()));
    try {
      addGeneratedFiles(moduleOut);
      addUserProvidedFiles(moduleOut, processDefinition);
    }
    finally {
      moduleOut.close();
    }
  }

  protected void addGeneratedFiles(ZipOutputStream moduleOut) throws IOException {
    FileUtil.zip(getWebDirectory(), moduleOut, DIR_WEB_INF);
    FileUtil.zip(getClassesDirectory(), moduleOut, DIR_WEB_INF + SEPARATOR + DIR_CLASSES);
  }

  protected void addUserProvidedFiles(ZipOutputStream moduleOut,
      BpelProcessDefinition processDefinition) throws IOException {
    FileDefinition fileDefinition = processDefinition.getFileDefinition();

    for (Iterator i = fileDefinition.getBytesMap().entrySet().iterator(); i.hasNext();) {
      Map.Entry entry = (Map.Entry) i.next();
      String fileName = (String) entry.getKey();

      if (fileName.startsWith(DIR_WEB_INF)) {
        byte[] fileContent = (byte[]) entry.getValue();
        if (fileContent == null)
          continue; // this entry denotes a directory, skip it

        moduleOut.putNextEntry(new ZipEntry(fileName));
        moduleOut.write(fileContent);
        moduleOut.closeEntry();
      }
    }
  }
}
