/*
 * Copyright 2010 Red Hat, Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.jboss.soa.dsp.deployer.tomcat;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;

import javax.wsdl.Definition;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.xml.namespace.QName;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.soa.dsp.EndpointMetaData;
import org.jboss.soa.dsp.ServiceEndpointReference;
import org.jboss.soa.dsp.deployer.ServiceDeployer;
import org.jboss.soa.dsp.server.ServerConfig;
import org.jboss.soa.dsp.ws.BuildProcessor;
import org.jboss.soa.dsp.ws.DeploymentBuilder;

public class TomcatServiceDeployer implements ServiceDeployer {

	private static final Log log = LogFactory.getLog(TomcatServiceDeployer.class);
	
	public ServiceEndpointReference deploy(EndpointMetaData metaData,
			Class<?> providerImpl, URL serviceUrl, ClassLoader classLoader,
			File deployment, ServerConfig serverConfig) throws Exception {
		
		String warPath=deployment.getAbsolutePath();
		
		File deployDir=getDeployDir(serviceUrl, serverConfig);
		
		log.debug("Copy "+warPath+" to "+deployDir);
		
		copyDirectory(new File(warPath), deployDir);
		
		ServiceEndpointReference ret=new ServiceEndpointReference(metaData.getEndpointId(),
						serviceUrl, metaData.getServiceName().getLocalPart(), warPath);
		
		return(ret);
	}
	
	protected File getDeployDir(URL serviceUrl, ServerConfig serverConfig) throws Exception {
		String context=serviceUrl.getPath();
		
		int ind=context.indexOf(1, '/');
		if (ind != -1) {
			context = context.substring(0, ind);
		}
		
		File deployDir=new File(serverConfig.getServerDeployDir(), context);
		
		return(deployDir);
	}

	protected void copyDirectory(File sourceLocation , File targetLocation) throws IOException {
        
		if (sourceLocation.isDirectory()) {
			if (!targetLocation.exists()) {
				targetLocation.mkdir();
			}
            
			String[] children = sourceLocation.list();
			for (int i=0; i<children.length; i++) {
				copyDirectory(new File(sourceLocation, children[i]),
						new File(targetLocation, children[i]));
			}
        } else {
            
        	InputStream in = new FileInputStream(sourceLocation);
        	OutputStream out = new FileOutputStream(targetLocation);
            
        	// Copy the bits from instream to outstream
        	byte[] buf = new byte[1024];
        	int len;
        	while ((len = in.read(buf)) > 0) {
        		out.write(buf, 0, len);
        	}
        	in.close();
        	out.close();
        }
	}
	
	public void undeploy(ServiceEndpointReference ref, ServerConfig serverConfig) throws Exception {
		
		File deployDir=getDeployDir(ref.getServiceUrl(), serverConfig);
		
		// Remove directory from the deployment folder
		if (deployDir != null) {
			delete(deployDir);
		}
	}
	
	protected void delete(File file) {
		if (file.isDirectory()) {
			File[] children=file.listFiles();
			
			for (int i=0; i < children.length; i++) {
				delete(children[i]);
			}
		}
		
		if (!file.delete()) {
			log.error("Failed to delete '"+file+"' when undeploying service");
		}
	}
	
	/**
	 * This method returns the optional build processor associated with the
	 * service deployer. This build processor can be used to performed any
	 * platform specific deployment archive modifications prior to it being
	 * deployed to the target platform.
	 * 
	 * @return The optional build procesor
	 */
	public BuildProcessor getBuildProcessor() {
		return(new TomcatBuildProcessor());
	}
	
	protected class TomcatBuildProcessor implements BuildProcessor {

		public void process(DeploymentBuilder builder) {
			String endpointId=builder.getEndpoint();
			String provider=(builder.getProvider() == null ? null : builder.getProvider().getClass().getName());
			java.util.List<String> urls=getURLs(builder.getWSDL());
			
			File sunjaxws=new File(builder.getWebInf(), "sun-jaxws.xml");

			createDescriptor(sunjaxws, endpointId, provider, urls);

			File web=new File(builder.getWebInf(), "web.xml");

			createDescriptor(web, endpointId, provider, urls);
		}
		
		protected void createDescriptor(File file, String endpointId, String provider, java.util.List<String> urls) {
			java.io.InputStream preio=TomcatBuildProcessor.class.getResourceAsStream("/files/pre-"+file.getName());
			java.io.InputStream midio=TomcatBuildProcessor.class.getResourceAsStream("/files/mid-"+file.getName());
			java.io.InputStream postio=TomcatBuildProcessor.class.getResourceAsStream("/files/post-"+file.getName());

			if (preio != null && midio != null && postio != null) {
				try {
					java.io.FileOutputStream fos=new java.io.FileOutputStream(file);

					byte[] b=new byte[preio.available()];
					preio.read(b);
					preio.close();
					
					String prestr=new String(b);
					prestr = prestr.replaceAll("ws.name", "ws"+endpointId);
					
					fos.write(prestr.getBytes());
					
					b = new byte[midio.available()];
					midio.read(b);
					midio.close();
					
					String str=new String(b);

					for (int i=0; i < urls.size(); i++) {
						java.net.URI uri=new java.net.URI(urls.get(i));
						String path=uri.getPath();
						
						int index=path.indexOf('/', 1);
						if (index == -1) {
							path = "/";
						} else {
							path = path.substring(index);
						}
						
						String midstr = str.replaceAll("ws.name", "ws"+endpointId+"-"+(i+1));
						midstr = midstr.replaceAll("ws.implementation", provider);
						midstr = midstr.replaceAll("ws.url.pattern", path);
						
						fos.write(midstr.getBytes());
					}
					
					b = new byte[postio.available()];
					postio.read(b);
					postio.close();
					
					fos.write(b);

					fos.close();
					
				} catch(Exception e) {
					log.error("Failed to create web service metadata", e);
				}		
			}
		}
		
		protected java.util.List<String> getURLs(File wsdl) {
			java.util.List<String> ret=new java.util.Vector<String>();
			
			if (wsdl != null) {
				try {
					javax.wsdl.factory.WSDLFactory fact=
						javax.wsdl.factory.WSDLFactory.newInstance();
					
					Definition defn=fact.newWSDLReader().readWSDL(wsdl.toURI().toString());
					
					java.util.Map<QName, Service> services=defn.getServices();
										
					//System.out.println("SERVICES="+services);
					
					java.util.Collection<Service> scol=services.values();
					
					for (Service service : scol) {						
						java.util.Map<QName,Port> ports=service.getPorts();

						java.util.Collection<Port> pcol=ports.values();
						
						for (Port port : pcol) {
							
							for (int i=0; i < port.getExtensibilityElements().size(); i++) {
								ExtensibilityElement ee=(ExtensibilityElement)port.getExtensibilityElements().get(i);
								
								if (ee instanceof javax.wsdl.extensions.soap.SOAPAddress) {
									ret.add(((javax.wsdl.extensions.soap.SOAPAddress)ee).getLocationURI());
								}
							}
						}
					}
					
				} catch(Exception e) {
					log.error("Failed to extract SOAP addresses from wsdl", e);
				}
			}
			
			return(ret);
		}
		
	}
}
