/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.internal.soa.esb.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

/**
 * LRU Reference Count cache.
 *
 * Cache implementation which caches the most recent caches, allowing entries to
 * be used by multiple users.  Requests for entries are tracked and counted,
 * returned to the cached set once no references remain.
 */
public class LRUReferenceCountCache<T, R extends LRUReferenceCountCacheResource<T>>
{
    /**
     * The unused entries.
     */
    private final Map<String, R> cachedResources = new HashMap<String, R>();
    /**
     * The list of cached resource entries.
     */
    private final LinkedHashSet<String> cachedResourceInsertion = new LinkedHashSet<String>();
    /**
     * The active entries.
     */
    private final Map<String, LRUReferenceCountCacheEntry<T, R>> active = new HashMap<String, LRUReferenceCountCacheEntry<T, R>>();
    /**
     * The cache size.
     */
    private final int lruCacheSize;
    /**
     * The logger instance.
     */
    private final Logger logger = Logger.getLogger(getClass()) ;

    public LRUReferenceCountCache(final int lruCacheSize)
    {
        this.lruCacheSize = lruCacheSize ;
    }

    public synchronized void clean()
    {
        cachedResourceInsertion.clear() ;
        final Collection<R> cachedResource = cachedResources.values() ;
        for(R resource: cachedResource)
        {
            closeResource(resource) ;
        }
        cachedResources.clear() ;
        if (active.size() > 0)
        {
            logger.warn("Active entries in cache") ;
        }
    }

    protected synchronized LRUReferenceCountCacheEntry<T, R> internalGet(final String resourceLocation)
    {
        final LRUReferenceCountCacheEntry<T, R> result ;
        final LRUReferenceCountCacheEntry<T, R> activeEntry = active.get(resourceLocation) ;
        if (activeEntry != null)
        {
            activeEntry.incCount() ;
            result = activeEntry ;
        }
        else
        {
            final R cachedResource = cachedResources.remove(resourceLocation) ;
            if (cachedResource != null)
            {
                cachedResourceInsertion.remove(resourceLocation) ;
                final LRUReferenceCountCacheEntry<T, R> entry = new LRUReferenceCountCacheEntry<T, R>(cachedResource, resourceLocation) ;
                active.put(resourceLocation, entry) ;
                result = entry ;
            }
            else
            {
                result = null ;
            }
        }
        return result ;
    }

    protected LRUReferenceCountCacheEntry<T, R> create(final String resourceLocation, R newResource)
    {
        final LRUReferenceCountCacheEntry<T, R> result ;
        boolean close = false ;
        
        synchronized (this)
        {
            final LRUReferenceCountCacheEntry<T, R> current = internalGet(resourceLocation) ;
            if (current == null)
            {
                final LRUReferenceCountCacheEntry<T, R> newEntry = new LRUReferenceCountCacheEntry<T, R>(newResource, resourceLocation) ;
                active.put(resourceLocation, newEntry) ;
                result = newEntry ;
            }
            else
            {
                close = true ;
                result = current ;
            }
        }
        if (close)
        {
            closeResource(newResource) ;
        }
        else
        {
            checkExpired() ;
        }
        return result ;
    }

    public void release(final LRUReferenceCountCacheEntry<T, R> entry)
    {
        boolean cached = false ;
        synchronized (this)
        {
            if (entry.decCount() == 0)
            {
                final String resourceLocation = entry.getResourceLocation() ;
                cachedResources.put(resourceLocation, entry.getCacheResource()) ;
                cachedResourceInsertion.add(resourceLocation) ;
                cached = true ;
            }
        }
        if (cached)
        {
            checkExpired() ;
        }
    }

    private void checkExpired()
    {
        final List<R> expiredResources ;
        synchronized (this)
        {
            final int activeSize = active.size() ;
            final int cachedSize = cachedResourceInsertion.size() ;
            final int expiredCount = (activeSize >= lruCacheSize ? cachedSize
                    : activeSize + cachedSize - lruCacheSize) ;
            if (expiredCount > 0)
            {
                expiredResources = new ArrayList<R>(expiredCount) ;
                final Iterator<String> iterator = cachedResourceInsertion.iterator() ;
                for (int count = 0; count < expiredCount; count++)
                {
                    final String expiredResourceLocation = iterator.next();
                    iterator.remove() ;
                    expiredResources.add(cachedResources.remove(expiredResourceLocation)) ;
                }
            }
            else
            {
                expiredResources = null ;
            }
        }
        if (expiredResources != null)
        {
            for (R resource : expiredResources)
            {
                closeResource(resource) ;
            }
        }
    }

    private void closeResource(final R resource)
    {
        try
        {
            resource.close() ;
        }
        catch (final Throwable th)
        {
            logger.warn("Unexpected exception cleaning resource", th) ;
        }
    }

    public static final class LRUReferenceCountCacheEntry<T, R extends LRUReferenceCountCacheResource<T>>
    {
        private final R resource;
        private final String resourceLocation;
        private int count = 1;
        
        LRUReferenceCountCacheEntry(final R resource, final String resourceLocation)
        {
            this.resource = resource ;
            this.resourceLocation = resourceLocation ;
        }

        R getCacheResource()
        {
            return resource ;
        }

        public T getResource()
        {
            return resource.getInstance() ;
        }

        public String getResourceLocation()
        {
            return resourceLocation ;
        }

        private int incCount()
        {
            return ++count ;
        }

        private int decCount()
        {
            return --count ;
        }
    }
}
