/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.routing.http;

import static org.junit.Assert.*;
import static org.junit.Assert.assertEquals;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.Before;
import org.junit.Test;

public class HttpRouterUnitTest
{
    @Before
    public void setUp()
    {
    }

    @Test
    public void testRouter_EmptyMappedHeaderList() throws ConfigurationException
    {    
        ConfigTree tree = new ConfigTree("EmptyMappedHeaderList");
        tree.setAttribute("endpointUrl", "http://foo.bar");
        tree.setAttribute("method", "post");
        HttpRouter router = new HttpRouter(tree);
        String[] headerList = router.getMappedHeaderList();
        assertEquals(0, headerList.length);
    }

    @Test
    public void testRouter_ValidMappedHeaderList() throws ConfigurationException
    {    
        ConfigTree tree = new ConfigTree("ValidMappedHeaderList");
        tree.setAttribute("endpointUrl", "http://foo.bar");
        tree.setAttribute("method", "post");
        tree.setAttribute("MappedHeaderList", "SOAPAction, Content-Type, Accept");
        HttpRouter router = new HttpRouter(tree);
        String[] headerList = router.getMappedHeaderList();
        assertEquals(3, headerList.length);

        List<String> headers = new ArrayList<String>();
        headers.add("SOAPAction");
        headers.add("Content-Type");        
        headers.add("Accept");    
        for(String h : headerList) {
            if(headers.contains(h)) {
                headers.remove(h);
            }
        }

        assertEquals(0, headers.size());
    }
    
    @Test
    public void testRouter_NullResponse() throws ConfigurationException, ActionProcessingException
    {    
        final ConfigTree tree = new ConfigTree("ValidMappedHeaderList");
        tree.setAttribute("endpointUrl", "http://foo.bar");
        tree.setAttribute("method", "post");
        tree.setAttribute("MappedHeaderList", "SOAPAction, Content-Type, Accept");
        
        HttpRouter router = new HttpRouter(tree);
        try
        {
            router.closeStream(null);
        } 
        catch (final IOException e)
        {
            fail(e.getMessage());
        }
    }

    @Test
    public void testRouter_NullBody() throws ConfigurationException, ActionProcessingException
    {    
        final ConfigTree tree = new ConfigTree("ValidMappedHeaderList");
        tree.setAttribute("endpointUrl", "http://foo.bar");
        tree.setAttribute("method", "post");
        tree.setAttribute("MappedHeaderList", "SOAPAction, Content-Type, Accept, If-Modified-Since");
        
        HttpRouter router = new HttpRouter(tree);
        try
        {
            router.readStream(null);
        }
        catch (final IOException e)
        {
            fail(e.getMessage());
        }
    }

    public static junit.framework.Test suite() {
        return new JUnit4TestAdapter(HttpRouterUnitTest.class);
    }
}
