/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */
package org.jboss.soa.esb.listeners.gateway;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.net.MalformedURLException;
import java.util.Arrays;
import java.util.List;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.internal.soa.esb.util.embedded.EmbeddableException;
import org.jboss.internal.soa.esb.util.embedded.ftp.FtpTestUtil;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.gateway.remotestrategies.ReadOnlyRemoteFileSystemStrategy;
import org.jboss.soa.esb.listeners.gateway.remotestrategies.cache.DeleteOnEvictTreeCacheListener;
import org.jboss.soa.esb.listeners.gateway.remotestrategies.cache.FtpFileCacheTestUtil;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.schedule.SchedulingException;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.testutils.HsqldbUtil;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Unit test for ReadOnlyRemoteGatewayListener that uses an embedded ftp server.
 * 
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>				
 *
 */
public class ReadOnlyRemoteGatewayListenerIntegrationTest extends RemoteGatewayListenerUnitTest
{
	
	@SuppressWarnings("unused")
	private static Logger log = Logger .getLogger( ReadOnlyRemoteGatewayListenerIntegrationTest.class );
	
	private static Object server;

        private static void registerHsqlDriver() {
            try { 
                Class.forName( "org.hsqldb.jdbcDriver" );
	    } catch (Exception e) { 
	        log.error(e);
	    }
	}
	       
	@BeforeClass
	public static void classSetup() throws EmbeddableException, ConfigurationException, GatewayException, RegistryException, MalformedURLException
	{
        MockCourierFactory.install();
        MockRegistry.install();
        MockRegistry.register(RemoteGatewayListenerUnitTest.SERVICE_CATEGORY, RemoteGatewayListenerUnitTest.SERVICE_NAME, new MockCourier(true));

        registerHsqlDriver();
        try {
            server = HsqldbUtil.startHsqldb(TestEnvironmentUtil.getUserDir() + "build" + File.separator + "hsqltestdb", "ftpcache");
        } catch (Exception e) {
            log.error(e);
            server = null;
        }
        
        ftpServer = createFtpServer();
		ftpServer.start();
		
		createWorkDirectories();
		configTree = createConfigTree();
		createRemoteGatewayListener( configTree );
	}

    @AfterClass
    public static void classCleanup() throws EmbeddableException, ConfigurationException, GatewayException, RegistryException, MalformedURLException
    {
        MockRegistry.uninstall();
        MockCourierFactory.uninstall();
    }

    @Test
	public void deleteFile() throws GatewayException 
	{
		File remoteFile = createAbsoluteFromFile();
				
		gatewayListener.deleteFile( remoteFile );
		assertTrue( "Remote file was deleted which should not be possible!",  testFile.exists() );
	}
	
	@Test
	public void getFileListFromRemoteDir() throws GatewayException
	{
		File[] fileListFromRemoteDir = gatewayListener.getFileList();
			
		assertNotNull ( fileListFromRemoteDir );
		assertTrue ( fileListFromRemoteDir.length > 0 );
			
		List<File> fileList = Arrays.asList( fileListFromRemoteDir );
		assertTrue( "The test file was not included in the List! " , fileList.contains( new File ( testFile.getName() )  ));
	}

	/*	
	@Test ( timeout=10000 )
	public void doRun_PostDir_Same_As_InputDir() throws ManagedLifecycleException, ConfigurationException, GatewayException, RegistryException, InterruptedException
	{
		setAbsoluteInputDir();
		setAbsolutePostDirToInputDir();
		boolean getCourierReturnValue = true;
		
		RemoteGatewayListenerMock mock = createAndStartListener ( configTree,  getCourierReturnValue );
		
		try {
			File inputDir =  ftpServer.getLocalInputDir();
			String doneFileName = testFile.getName();
			
			//waitForFile( inputDir, doneFileName );
			//stopdoRun( mock );
			try {
				mock.onSchedule();
			} catch (SchedulingException e) {
				fail(e.getMessage());
			}
			
			File processedFile = new File ( inputDir,  doneFileName );
			try
			{
				assertTrue ( processedFile.exists() );
			}
			finally
			{
				FtpTestUtil.deleteFile( processedFile );
			}
		} finally {
			mock.destroy();
		}
	}
	*/
	
	/**
	 * This is a "negtive" test. It verifies that the file was not
	 * renamed. The name of the method is kept so that it overrides 
	 * the super classes method.
	 */
	@Test
	public void renameFile_In_Same_Directory() throws GatewayException 
	{
	/* 
		File from = createAbsoluteFromFile();
		File to = createAbsoluteToFile();
			
		gatewayListener.renameFile( from, to );
			
		renamedFile = new File ( ftpServer.getLocalInputDir(), to.getName() );
		assertFileExists( testFile );
		assertFalse( renamedFile.exists() );
		*/
	}
	
	
	@Test 
	public void getNewInstanceNegative() 
	{
		ReadOnlyRemoteGatewayListener listener = (ReadOnlyRemoteGatewayListener) gatewayListener;
		try
		{
			listener.getNewInstanceOf( null );
			fail ( "an IllegalArgumentException should have been thrown" );
		} 
		catch ( Exception e)
		{
			assertTrue ( e instanceof IllegalArgumentException );
		}
		
		try
		{
			listener.getNewInstanceOf( "some.package.class" );
			fail ( "a ConfigurationException should have been thrown" );
		} 
		catch ( Exception e)
		{
			assertTrue ( e instanceof ConfigurationException );
		}
	}
	
	@Test 
	public void getNewInstance() throws ConfigurationException
	{
		ReadOnlyRemoteGatewayListener listener = (ReadOnlyRemoteGatewayListener) gatewayListener;
		ReadOnlyRemoteFileSystemStrategy instance = listener.getNewInstanceOf( ReadOnlyRemoteFileSystemStrategy.class.getName() );
		assertNotNull( "The instance should not be null!", instance );
	}
	
	/**
	 * Not a valid test for this impl
	 */
	@Override
	public void renameFile_In_Different_Directory() throws GatewayException { }
	
	/**
	 * Not a valid test for this impl
	 */
	@Override
	public void doRun_PostDir_Different_From_InputDir() throws ManagedLifecycleException, ConfigurationException, GatewayException, RegistryException, InterruptedException { }
	
	/**
	 * Not a valid test for this impl
	 */
	@Override
	public void doRun_Negative() throws ManagedLifecycleException, ConfigurationException, GatewayException, RegistryException, InterruptedException {}
	
	/**
	 * Override createConfigTree and add the ReadOnlyFileSystem strategy.
	 * 
	 * @return
	 */
	public static ConfigTree createConfigTree()
	{
		ConfigTree configTree = RemoteGatewayListenerUnitTest.createConfigTree();
		configTree.setAttribute( ReadOnlyRemoteGatewayListener.REMOTE_FILE_SYSTEM_STRATEGY_CLASS, ReadOnlyRemoteFileSystemStrategy.class.getName() );
		configTree.setAttribute( ReadOnlyRemoteGatewayListener.REMOTE_FILE_SYSTEM_STRATEGY_CONFIG_FILE, FtpFileCacheTestUtil.getCacheConfigFile() );
		configTree.setAttribute( ReadOnlyRemoteGatewayListener.REMOTE_FILE_SYSTEM_STRATEGY_CACHE_LISTENER, DeleteOnEvictTreeCacheListener.class.getName() );
		return configTree;
	}
	
	protected static void createRemoteGatewayListener( ConfigTree configTree )
	throws ConfigurationException, RegistryException, GatewayException
	{
		gatewayListener = new ReadOnlyRemoteGatewayListener( configTree );
	}
	
	/**
	 * Creates a RemoteGatewayListenerMock instance with the passed in arguments, and then starts
	 * the ReadOnlyRemoteGatewayListener's doRun() method in a new Thread.
	 * 
	 * @param configTree The configTree to use to configure the RemoteGatewayListenerMock object
	 * @param returnValueForCourier	the return value for the getCourier method of the RemoteGatewayListenerMock object
	 * @return RemoteGatewayListenerMock Mock impl of a RemoteGatewayListener.
	 */
	protected RemoteGatewayListenerMock createAndStartListener ( ConfigTree configTree,  boolean returnValueForCourier ) throws ConfigurationException, GatewayException, RegistryException
	{
		final RemoteGatewayListenerMock mock = new RemoteGatewayListenerMock( configTree );
        try {
            mock.initialise();
        } catch (ManagedLifecycleException e) {
            fail(e.getMessage());
        }

        if(returnValueForCourier) {
            MockCourierFactory.courierException = null;
        } else {
            MockCourierFactory.courierException = new CourierException("Test generated mock exception.");
        }

        return mock;
		
	}
	
	/**
	 *  Mock impl of RemoteGatewayListener. 
	 */
	class RemoteGatewayListenerMock extends ReadOnlyRemoteGatewayListener
	{
		private boolean finishedOneRun;
		
		/**
		 * Sole constructor
		 * 
		 * @param configTree						configTree used to configure this RemoteGatewayListener
		 * 
		 */
		public RemoteGatewayListenerMock ( ConfigTree configTree) throws ConfigurationException, GatewayException, RegistryException
		{
			super ( configTree );
        }

		public boolean hasFinishedOneRun()
		{
			return finishedOneRun;
		}
	}
	
	/*
	 * Just here to help Ant to find annotated test.
	 */
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( ReadOnlyRemoteGatewayListenerIntegrationTest.class);
	}
	
}
