/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2010
 */
package org.jboss.internal.soa.esb.services.rules;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import org.drools.runtime.Channel;
import org.jboss.soa.esb.Configurable;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.rules.RuleInfo;
import org.jboss.soa.esb.services.rules.ServiceChannel;
import org.jboss.soa.esb.services.rules.StatefulRuleInfo;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

/**
 * RuleServiceCallHelperUnitTest.
 * 
 * @author dward at jboss.org
 */
public class RuleServiceCallHelperUnitTest
{
	
	private static final String xml = new StringBuffer()
		.append("<action>")
		.append("    <send-to channel-name='channel_1' channel-class='" + TestChannel.class.getName() + "'>")
		.append("        <property name='expected' value='test_1'/>")
		.append("    </send-to>")
		.append("    <send-to channel-name='channel_2' channel-class='" + TestChannel.class.getName() + "'>")
		.append("        <property name='expected' value='test_2'/>")
		.append("    </send-to>")
		.append("    <send-to channel-name='channel_2' channel-class='" + TestChannel.class.getName() + "'>")
		.append("        <property name='expected' value='test_2'/>")
		.append("    </send-to>")
		.append("    <send-to channel-name='channel_2' channel-class='" + TestChannel.class.getName() + "'>")
		.append("        <property name='expected' value='test_2'/>")
		.append("    </send-to>")
		.append("    <send-to channel-name='channel_3' service-category='category_3' service-name='service_3'/>")
		.append("</action>")
		.toString();
	
	public static class TestChannel implements Channel, Configurable {
		private static int count = 0;
		private String expected = null;
		public TestChannel() {}
		public void setConfiguration(ConfigTree config) throws ConfigurationException {
			for (ConfigTree child : config.getChildren("property")) {
				if ("expected".equals(child.getAttribute("name"))) {
					expected = child.getRequiredAttribute("value");
					break;
				}
			}
		}
		public void send(Object actual) {
			assertEquals(expected, actual);
			count++;
		}
	}
	
	@Before
	@After
	public void reset()
	{
		TestChannel.count = 0;
	}
	
    @Test
    public void testGetChannels() throws Exception
    {
    	ConfigTree config = ConfigTree.fromXml(xml);
    	
    	Map<String,Channel> channel_map = RuleServiceCallHelper.getChannels(config);
    	assertNotNull(channel_map);
    	assertTrue(channel_map.size() == 3);
    	
    	Channel channel_1 = channel_map.get("channel_1");
    	assertNotNull(channel_1);
    	channel_1.send("test_1");
    	assertEquals(1, TestChannel.count);
    	
    	Channel channel_2 = channel_map.get("channel_2");
    	assertNotNull(channel_2);
    	channel_2.send("test_2");
    	assertEquals(4, TestChannel.count);
    	
    	Channel channel_3 = channel_map.get("channel_3");
    	assertNotNull(channel_3);
    	assertTrue(channel_3 instanceof ServiceChannel);
    	Service service_3 = ((ServiceChannel)channel_3).getService();
    	assertNotNull(service_3);
    	assertEquals("category_3", service_3.getCategory());
    	assertEquals("service_3", service_3.getName());
    }
    
    @Test
    public void testRunChannels() throws Exception
    {
    	// create the statefulRuleInfo (including the channels)
    	ConfigTree config = ConfigTree.fromXml(xml);
    	Map<String,Channel> channel_map = RuleServiceCallHelper.getChannels(config);
    	channel_map.remove("channel_3");
		RuleInfo ruleInfo = new RuleInfoBuilder("RuleServiceCallHelper.drl").channels(channel_map).build();
		StatefulRuleInfo statefulRuleInfo = new StatefulRuleInfoImpl(ruleInfo, true, false);
		
		// create the message
		Message message = MessageFactory.getInstance().getMessage();
		
		// execute the rules (which will execute the channels)
		new DroolsRuleService().executeStatefulRules(statefulRuleInfo, message);
		assertEquals(4, TestChannel.count);
    }
    
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(RuleServiceCallHelperUnitTest.class);
    }

}
