/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.deployers.mc.as6;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.jboss.logging.Logger;
import org.jboss.deployment.DeploymentException;
import org.jboss.internal.soa.esb.listeners.war.Filter;
import org.jboss.internal.soa.esb.listeners.war.Servlet;
import org.jboss.internal.soa.esb.listeners.war.WebDeploymentArchive;
import org.jboss.internal.soa.esb.listeners.war.WebModel;
import org.jboss.internal.soa.esb.publish.ContractReferencePublisher;
import org.jboss.internal.soa.esb.util.JBossDeployerUtil;
import org.jboss.internal.soa.esb.webservice.ESBContractGenerator;
import org.jboss.internal.soa.esb.webservice.ESBResponseFilter;
import org.jboss.internal.soa.esb.webservice.ESBServiceContractReferencePublisher;
import org.jboss.internal.soa.esb.webservice.ESBServiceEndpointInfo;
import org.jboss.internal.soa.esb.webservice.JAXWSProviderClassGenerator;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.listeners.config.model.ModelAdapter;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.util.DeploymentArchive;

import org.jboss.soa.esb.listeners.config.ModelUtil;
import org.jboss.soa.esb.listeners.config.WebserviceInfo;

/**
 * A builder that builds EBWS/HTTP Gateway web archives
 *
 * @author <a href="mailto:mageshbk@jboss.com">Magesh Kumar B</a>
 * @version $Revision: 1.0 $
 */
public class WebGatewayBuilder
{
    private Logger log = Logger.getLogger(WebGatewayBuilder.class);

    private String deploymentName;
    private File esbWarFiles;
    private File esbArchive;
    private ClassLoader localCl;
    private List<ContractReferencePublisher> publishers;
    private List<Servlet> servlets;
    private ModelAdapter model;

    public WebGatewayBuilder(final File esbWarFiles, final File esbArchive, final String deploymentName, final ClassLoader localCl, final ModelAdapter model)
    {
       this.esbWarFiles = esbWarFiles;
       this.esbArchive = esbArchive;
       this.deploymentName = deploymentName;
       this.localCl = localCl;
       this.model = model;
    }

    public void setPublishers(final List<ContractReferencePublisher> publishers)
    {
        this.publishers = publishers;
    }
    
    public List<ContractReferencePublisher> getPublishers()
    {
       return publishers;
    }
    
    public void setServlets(final List<Servlet> servlets)
    {
       this.servlets = servlets;
    }
    
    public List<Servlet> getServlets()
    {
       return servlets;
    }

    public File build() throws ConfigurationException, DeploymentException
    {
        File war = null;
        WebDeploymentArchive webDeployment = new WebDeploymentArchive(getESBDeploymentName(), getESBWarFileName());
        WebModel webModel = webDeployment.getWebModel();

        // Set the global security domain and global security method.
        // These setting are shared for all http-providers and EBWSs 
        // in a jboss-esb.xml file. 
        webModel.setAuthDomain(model.getAuthDomain());
        webModel.setAuthMethod(model.getAuthMethod());

        // Add the EBWS components...
        createWebserviceWars(webDeployment);

        // Add the web deployments info to the WebModel...
        ModelUtil.updateWebModel(deploymentName, ModelUtil.getListenerGroups(model), webModel);

        // Add a sub-deloyment for the web model...
        this.servlets = webModel.getServlets();
        if(!servlets.isEmpty()) {
            
            // Finalize the webDeployment...
            webDeployment.finalizeArchive();

            // Now create the web deployment...
            war = createGatewayTempDeployment(webDeployment);

        }
        return war;
    }

    private void createWebserviceWars(WebDeploymentArchive webDeployment) throws DeploymentException
    {
        this.publishers = new ArrayList<ContractReferencePublisher>();
        try
        {
            final List<WebserviceInfo> endpointServices =  model.getWebserviceServices();
            if (endpointServices != null)
            {
                if (endpointServices.size() > 0)
                {
                    final JAXWSProviderClassGenerator generator = new JAXWSProviderClassGenerator();
                    
                    for(WebserviceInfo webserviceInfo: endpointServices)
                    {
                        final Service service = webserviceInfo.getService();
 
                        // Copy all schemas to the wsdl directory to support imports.
                        final Map<String, String> schemasMap = JBossDeployerUtil.getSchemas(esbArchive);
                        final String wsdlDir = "WEB-INF/wsdl/" + service.getCategory().replace('/', '_') + "/";
                        for (Entry<String, String> schemaEntry : schemasMap.entrySet())
                        {
                            webDeployment.addEntry(wsdlDir + schemaEntry.getKey(), schemaEntry.getValue().getBytes());
                            log.debug("Added schema " + wsdlDir + schemaEntry.getKey());
                        }
                        
                        final ESBServiceEndpointInfo serviceInfo = new ESBServiceEndpointInfo(webserviceInfo);
                        final String wsdl = ESBContractGenerator.generateWSDL(webserviceInfo, serviceInfo, new WebDeploymentClassLoader(localCl, schemasMap));
                        webDeployment.addEntry(serviceInfo.getWSDLFileName(), wsdl.getBytes("UTF-8"));
 
                        final String handlers = JBossDeployerUtil.getHandlers(serviceInfo);
                        final boolean includeHandlers = (handlers != null);
                        if (includeHandlers)
                        {
                            final String wsHandlerName = "WEB-INF/classes/" + serviceInfo.getPackageName().replace('.', '/') + "/esb-jaxws-handlers.xml";
                            webDeployment.addEntry(wsHandlerName, handlers.getBytes("UTF-8"));
                        }
                        
                        final byte[] wsClass = generator.generate(deploymentName, service.getCategory(),
                            service.getName(), serviceInfo, includeHandlers);
                        final String wsClassName = serviceInfo.getClassName().replace('.', '/') + ".class";
                        webDeployment.addEntry("WEB-INF/classes/" + wsClassName, wsClass);
 
                        // Servlet....
                        final Servlet servlet = new Servlet(serviceInfo.getServletName(), serviceInfo.getClassName(), webDeployment.getWebModel());
                        servlet.getUrlMappings().add("/ebws" + serviceInfo.getServletPath());
 
                        // Filter...
                        // At the moment we only need the filter to overcome a bug in JBossWS re In-Only endpoints, so
                        // we only include for one way services...
                        if(serviceInfo.isOneWay()) {
                            new Filter(serviceInfo.getServletName() + "_Filter", ESBResponseFilter.class.getName(), servlet);
                        }
 
                        final ContractReferencePublisher publisher = new ESBServiceContractReferencePublisher(service, webserviceInfo.getDescription(), serviceInfo.getServletName());
                        publishers.add(publisher);
                    }
                }
            }
        }
        catch (final Exception ex)
        {
            throw new DeploymentException("Failed to create webservice artifact", ex);
        }
    }

    private File createGatewayTempDeployment(DeploymentArchive deploymentArchive) throws ConfigurationException
    {

        File gatewayDeployFile = createTempDeploymentFile(esbWarFiles, getESBWarFileName(), true);
        FileOutputStream fileOutStream;

        try {
            fileOutStream = new FileOutputStream(gatewayDeployFile);
        } catch (FileNotFoundException e) {
            throw new ConfigurationException("Unexpected exception.", e);
        }

        try {
            deploymentArchive.toOutputStream(fileOutStream);
        } catch (IOException e) {
            throw new ConfigurationException("Error writing deployment archive '" + gatewayDeployFile.getAbsolutePath() + "'.", e);
        } finally {
            try {
                fileOutStream.close();
            } catch (IOException e) {
                log.warn("Error closing deployment archive stream '" + gatewayDeployFile.getAbsolutePath() + "'", e);
            }
        }

        return gatewayDeployFile;
    }

    private File createTempDeploymentFile(final File tempDir, final String archiveName, final boolean createDir) throws ConfigurationException
    {
        if (!tempDir.exists())
        {
            if (!createDir)
            {
                return null;
            }
            tempDir.mkdir();
        }

        File file = new File(tempDir, archiveName);
        if(file.exists()) {
            log.debug("Deployment archive '" + archiveName + "' already exists.  Overwriting!!");
        }

        return file;
    }

    private String getESBDeploymentName()
    {
        final String esbName = deploymentName;
        final int lastSeparator = esbName.lastIndexOf('.');
        return ((lastSeparator >= 0) ? esbName.substring(0, lastSeparator) : esbName);
    }
    
    private String getESBWarFileName()
    {
        final String esbName = deploymentName;
        final int lastSeparator = esbName.lastIndexOf('.');
        return ((lastSeparator >= 0) ? esbName.substring(0, lastSeparator) : esbName) + "-web" +".war";
    }

    private class WebDeploymentClassLoader extends ClassLoader
    {
           
       private Map<String, String> schemas;

       WebDeploymentClassLoader(final ClassLoader parent, final Map<String, String> schemas)
       {
          super(parent);
          this.schemas = schemas;
       }

       @Override
       public InputStream getResourceAsStream (final String name)
       {
           if (schemas != null)
           {
               String schema = schemas.get(name);
               if (schema != null)
                   return new ByteArrayInputStream(schema.getBytes());
           }
          
           return super.getResourceAsStream(name);
       }
    }
}
