package org.jboss.internal.soa.esb.addressing.helpers;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.soa.esb.MarshalException;
import org.jboss.soa.esb.UnmarshalException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.PortReference;
import org.jboss.soa.esb.addressing.XMLUtil;
import org.jboss.soa.esb.addressing.PortReference.Extension;
import org.jboss.soa.esb.addressing.eprs.*;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.List;

public class EPRHelper
{
	public static final String EPR_TYPE = EPR.EPR_TYPE;
	
        private static final QName HEADER = new QName("header") ;


        public static void toXML(final XMLStreamWriter out, final QName name,
            final EPR epr)
            throws XMLStreamException
        {
            PortReferenceHelper.toXML(out, name, epr.getAddr()) ;
        }

        public static EPR fromXML(XMLStreamReader in)
            throws XMLStreamException
        {
            final PortReference portReference = PortReferenceHelper.fromXML(in) ;
            return getSpecificEPR(new EPR(portReference)) ;
        }
        
	/**
	 * Create a string version of the XML representation for this EPR. If the
	 * EPR is a specific type (e.g., JMSEpr) then that type information will
	 * also be encoded.
	 * 
	 * @param epr
	 * @return
	 * @throws MarshalException 
	 */

	public static final String toXMLString(EPR epr) throws MarshalException
	{
		return toXMLString(epr, XMLUtil.QNAME_FROM_TAG);
	}

	/**
	 * Create a string version of the XML representation for this EPR. If the
	 * EPR is a specific type (e.g., JMSEpr) then that type information will
	 * also be encoded. Specify whether this is a To node in the Call.
	 * 
	 * @param epr
	 * @return
	 * @throws MarshalException 
	 */

	public static final String toXMLString(EPR epr, QName name) throws MarshalException
	{
		AssertArgument.isNotNull(epr, "epr");
		AssertArgument.isNotNull(name, "name");

		try
		{
		    final StringWriter writer = new StringWriter() ;
		    final XMLStreamWriter out = XMLHelper.getXMLStreamWriter(writer) ;
		    
		    if (XMLUtil.QNAME_TO_TAG.equals(name))
		    {
		        final String origURI = StreamHelper.writeStartElement(out, HEADER) ;
                        toXML(out, name, epr) ;
		        StreamHelper.writeEndElement(out, HEADER.getPrefix(), origURI) ;
		    }
		    else
		    {
		        toXML(out, name, epr) ;
		    }
		    out.flush() ;
		    return writer.toString() ;
		}
		catch (XMLStreamException xmlse)
		{
                    throw new MarshalException("Failed to serialise EPR.", xmlse);
		}
	}

	/**
	 * Get the EPR from the string representation. If the EPR was a specific
	 * type (e.g., JMSEpr) then it will be returned as an instance of the
	 * appropriate class.
	 * 
	 * @param xml
	 * @return
	 * @throws UnmarshalException 
	 */

	public static final EPR fromXMLString(String xml) throws UnmarshalException
	{
		AssertArgument.isNotNull(xml, "xml");

		try
		{
		    final StringReader reader = new StringReader(xml) ;
		    final XMLStreamReader in = XMLHelper.getXMLStreamReader(reader) ;
		    StreamHelper.skipToStartElement(in) ;
		    if (HEADER.equals(in.getName()))
		    {
		        StreamHelper.skipToNextStartElement(in) ;
		    }
		    
		    return fromXML(in) ;
		}
		catch (final XMLStreamException xmlse)
		{
			throw new UnmarshalException("Unable to parse EPR XML.", xmlse);
		}
	}

	public final static EPR getSpecificEPR(EPR epr)
	{
		String eprType = epr.getAddr().getExtensionValue(EPR_TYPE);

		if (eprType != null)
		{
			/*
			 * Do not re-order.
			 */

			if (eprType.equals(InVMEpr.type().toString()))
				return new InVMEpr(epr);
			else if (eprType.equals(EmailEpr.type().toString()))
				return new EmailEpr(epr);
			else if (eprType.equals(FTPSEpr.type().toString()))
				return new FTPSEpr(epr);
			else if (eprType.equals(SFTPEpr.type().toString()))
				return new SFTPEpr(epr);
			else if (eprType.equals(HTTPEpr.type().toString()))
				return new HTTPEpr(epr);
			else if (eprType.equals(JDBCEpr.type().toString()))
				return new JDBCEpr(epr);
			else if (eprType.equals(HibernateEpr.type().toString()))
				return new HibernateEpr(epr);
			else if (eprType.equals(JMSEpr.type().toString()))
				return new JMSEpr(epr);
			else if (eprType.equals(FTPEpr.type().toString()))
				return new FTPEpr(epr);
			else if (eprType.equals(FileEpr.type().toString()))
				return new FileEpr(epr);
            else if (eprType.equals(LogicalEPR.type().toString()))
                return new LogicalEPR(epr);
			else
				return epr;
		} else {
			return epr;
		}
	}

    /**
     * Create a copy of the EPR, including the specified extensions.
     * @param epr The current EPR.
     * @param extensions The extensions to add.
     * @return The EPR copy.
     */
    public static EPR copyEPR(final EPR epr, final List<Extension> extensions)
    {
        final EPR copyEPR = new EPR(epr.getAddr().copy()) ;
        final PortReference addr = copyEPR.getAddr() ;
        if (extensions != null)
        {
            for(Extension extension: extensions)
            {
                if (!(EPR_TYPE.equals(extension.getTag()) &&
                    XMLUtil.JBOSSESB_NAMESPACE_URI.equals(extension.getURI()) &&
                    XMLUtil.JBOSSESB_PREFIX.equals(extension.getPrefix())))
                {
                    addr.addExtension(extension) ;
                }
            }
        }
        return copyEPR ;
    }
}