/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.notification;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.List;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;

/**
 * NotifyFTPList extends NotifyFTP and adds the ability to take
 * a single file name or list of file names located in the ESB Message object.
 * <p/>
 * The file(s) in the Message object should contain a list of files (full paths). This list 
 * will be iterated  over and every file in the list will be sent to </br>
 * the configured destination FTP server directory.
 * <p/>
 * Example configuration:
 * <pre>
 *&lt;action name="notificationAction" class="org.jboss.soa.esb.actions.Notifier"&gt;
 *    &lt;property name="okMethod" value="notifyOK"/&gt;
 *        &lt;property name="notification-details"&gt;
 *            &lt;NotificationList type="ok"&gt;
 *                &lt;target class="NotifyFTPList"&gt;
 *                    &lt;ftp URL="ftp://username:password@localhost/outputdir"
 *                        filename="{org.jboss.soa.esb.gateway.file}"&gt;
 *                        listFiles="false"
 *                        deletelistFile="false"
 *                &lt;/target&gt;
 *            &lt;/NotificationList&gt;
 *        &lt;/property&gt;
 *&lt;/action&gt;
 * </pre>
 * Description of configuration properties:
 * <ul>
 * <li><code>class </code> is name of this class
 * <li><code>URL </code> is the remote FTP server
 * <li><code>filename </code> see {@link NotifyFTP} for details about this configuration property
 * <li><code>listFiles </code> whether the file(s) located in the message body contain paths to files that are to be sent or simply point to files that contain lists for files to be sent.
 * <li><code>deleteListFile </code> indicates if the listFile should be deleted after all items items in it have been sent.
 * </ul>
 *  
 * @author <a href="mailto:daniel.bevenius@gmail.com">Daniel Bevenius</a>			
 * @since 4.3
 */
public class NotifyFTPList extends NotifyFTP
{
	private Logger log = Logger.getLogger( NotifyFTPList.class );
	
	public static final String FILE_LOCATION = NotifyFTPList.class.getName() + "#filename";
	
	/*
     * Indicated whether the file(s) located in the message body contain paths to files 
     * that are to be sent or simply point to files that contain lists for files to be sent.
	 */
	private boolean listFiles;

	private boolean deleteListFile;

	public NotifyFTPList(ConfigTree config) throws NotificationException
	{
		super( config );
		ConfigTree ftpConfig = getFtpConfig();
		listFiles = ftpConfig.getBooleanAttribute( "listFiles", false );
		deleteListFile = ftpConfig.getBooleanAttribute( "deleteListFile", false );
	}

	/**
	 * If the Message body contains a List<String> of files
	 */
	@Override
	public void sendNotification( Message message ) throws NotificationException
	{
		Object object = message.getBody().get();
		if ( object instanceof List )
		{
			List<String> files = (List<String>) object;
			for (String fileName : files)
			{
				send ( fileName, message );
			}
		}
		else if ( object instanceof String )
		{
			String fileName = (String) object;
			send ( fileName, message );
		}
	}
	
	private void send ( final String fileName, final Message message ) throws NotificationException
	{
		File file = new File( fileName );
		if ( listFiles )
		{
			sendFiles( file, message );
		}
		else
		{
			sendFile( file, message );
		}
	}
	
	private void sendFile( final File file, final Message message ) throws NotificationException
	{
		log.debug( "Sending file [" + file.getAbsolutePath() + "]" );
		message.getBody().add( FILE_LOCATION, file );
		send ( message );
	}
	
	private void sendFiles( File listFile, final Message message ) throws NotificationException
	{
		BufferedReader reader = null;
		try
		{
    		reader = new BufferedReader( new FileReader( listFile ) );
    		String fileName = null;
    		log.debug( "Sending from listFile [" + listFile.getAbsolutePath() + "]" );
    		while ( (fileName = reader.readLine() ) != null )
    		{
    			message.getBody().add( FILE_LOCATION, new File ( fileName ) );
    			send ( message );
    		}
    		log.debug( "Done sending from listFile [" + listFile.getAbsolutePath() + "]" );
    	} 
		catch (FileNotFoundException e)
		{
    		throw new NotificationException("Could not find file [" + listFile.getAbsolutePath() + "]", e );
		} 
		catch (IOException e)
		{
    		throw new NotificationException("Could not read from file [" + listFile.getAbsolutePath() + "]", e );
		}
		finally
		{
			if ( reader != null )
			{
				try
				{
					reader.close();
				} 
				catch (IOException e)
				{
					log.error( "IOException while trying to close reader :", e );
				}
			}
		}
		if ( deleteListFile )
		{
			boolean deleted = listFile.delete();
			if ( !deleted )
			{
				log.warn( "Could not delete listFile [" + listFile.getAbsolutePath() + "]" );
			}
		}
	}

	protected void send( final Message message ) throws NotificationException
	{
		super.sendNotification( message );
	}
	
	@Override
	protected File getFileToSend( Message message ) 
	{
		return (File) message.getBody().get( FILE_LOCATION ) ;
	}
	
	@Override
	protected String getFileName( Message message ) throws NotificationException 
	{
		File file = (File) message.getBody().get( FILE_LOCATION ) ;
		if ( file == null )
		{
			throw new NotificationException( "Could not find a file in the file location" );
		}
		return file.getName();
	}

}
