/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security;

import java.io.IOException;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.security.plugins.FilePassword;

/**
 * Util class that can read a password from a specified file.
 * </p>
 * This class simply delegated to {@link FilePassword} which performs the
 * real work.
 * 
 * <h3>Creating a password file</h3>
 * Go to the conf directory of your jboss server instance  (eg: default/conf)
 * java -cp ../lib/jbosssx.jar org.jboss.security.plugins.FilePassword welcometojboss 13 testpass passwordFile
 * <lu>
 *  <li>welcometojboss Is the salt value.<li>
 *  <li>testpass Is the clear text password you want to protect.<li>
 *  <li>passwordFile Is the name of the file which will contain the encrypted password.<li>
 * </lu>
 * 
 * Note that this is security by obscurity in that the password is not store
 * in plaintext, but it can be recovered by simply using the code from this class.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * @since 4.7
 */
public final class PasswordUtil
{
	private final PasswordHandler passwordHandler ;
	
    public PasswordUtil(final String passwordFile)
    {
        AssertArgument.isNotNull(passwordFile, "passwordFile");
        passwordHandler = PasswordPluginManager.getManager().passwordHandler(passwordFile) ;
        if (passwordHandler == null)
        {
        	throw new IllegalArgumentException("Could not determine password handler for " + passwordFile) ;
        }
    }
    
    public char[] getPassword() throws IOException
    {
        return passwordHandler.getPassword().toCharArray() ;
    }
    
    public String getPasswordAsString() throws IOException
    {
        return passwordHandler.getPassword() ;
    }
    
    public static boolean isPasswordFile(final String passwordFile)
    {
    	return (PasswordPluginManager.getManager().isPasswordSupported(passwordFile)) ;
    }
}
