/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.smooks.resource;

import java.io.IOException;
import java.io.InputStream;
import java.util.IdentityHashMap;
import java.util.Iterator;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.lifecycle.LifecyclePriorities;
import org.jboss.soa.esb.lifecycle.LifecycleResource;
import org.jboss.soa.esb.lifecycle.LifecycleResourceException;
import org.jboss.soa.esb.lifecycle.LifecycleResourceFactory;
import org.milyn.Smooks;
import org.milyn.container.standalone.StandaloneApplicationContext;
import org.milyn.resource.URIResourceLocator;
import org.xml.sax.SAXException;

/**
 * Smooks resource tied to the lifecycle.
 */
public class SmooksResource
{
    /**
     * The logger for this class.
     */
    private static Logger LOGGER = Logger.getLogger(SmooksResource.class);
    
    /**
     * The lifecycle resource factory.
     */
    private static final LifecycleResourceFactory<IdentityHashMap<Smooks, Smooks>> lifecycleSmooksResourceFactory = new SmooksResourceFactory() ;
    /**
     * Lifecycle smooks resources.
     */
    private static final LifecycleResource<IdentityHashMap<Smooks, Smooks>> lifecycleSmooksResource =
        new LifecycleResource<IdentityHashMap<Smooks, Smooks>>(lifecycleSmooksResourceFactory, LifecyclePriorities.SMOOKS_RESOURCE_PRIORITY) ;
    
    private SmooksResource()
    {
    }

    /**
     * Create a smooks instance.
     * @return The smooks instance.
     * @throws LifecycleResourceException For errors creating the lifecycle resource.
     */
    public static Smooks createSmooksResource()
        throws LifecycleResourceException
    {
        final StandaloneApplicationContext applicationContext = new DeploymentApplicationContext(true) ;
        final Smooks smooks = new Smooks(applicationContext) ;
        lifecycleSmooksResource.getLifecycleResource().put(smooks, smooks) ;
        
        return smooks;
    }

    /**
     * Create a smooks instance using the specified configuration.
     * @param config The configuration representing the smooks configuration.
     * @return The smooks instance.
     * @throws SAXException For errors parsing the smooks configuration.
     * @throws IOException For errors accessing the smooks configuration.
     * @throws LifecycleResourceException For errors creating the lifecycle resource.
     */
    public static Smooks createSmooksResource(final String config)
        throws SAXException, IOException, LifecycleResourceException
    {
        final URIResourceLocator resourceLocator = new URIResourceLocator() ;
        resourceLocator.setBaseURI(URIResourceLocator.extractBaseURI(config)) ;

        final StandaloneApplicationContext applicationContext = new DeploymentApplicationContext(true) ;
        applicationContext.setResourceLocator(resourceLocator);

        final Smooks smooks = new Smooks(applicationContext) ;
        smooks.addConfigurations(config) ;

        lifecycleSmooksResource.getLifecycleResource().put(smooks, smooks) ;
        
        return smooks;
    }

    /**
     * Create a smooks instance using the specified input stream.
     * @param is The input stream representing the smooks configuration.
     * @return The smooks instance.
     * @throws SAXException For errors parsing the smooks configuration.
     * @throws IOException For errors accessing the smooks configuration.
     * @throws LifecycleResourceException For errors creating the lifecycle resource.
     */
    public static Smooks createSmooksResource(final InputStream is)
        throws SAXException, IOException, LifecycleResourceException
    {
        final StandaloneApplicationContext applicationContext = new DeploymentApplicationContext(true) ;
        applicationContext.setResourceLocator(new URIResourceLocator()) ;

        final Smooks smooks = new Smooks(applicationContext) ;
        smooks.addConfigurations(is) ;
        
        lifecycleSmooksResource.getLifecycleResource().put(smooks, smooks) ;
        
        return smooks;
    }

    /**
     * Close the smooks resource.
     * @param smooks The smooks resource.
     */
    public static void closeSmooksResource(final Smooks smooks)
    {
        try
        {
            smooks.close() ;
        }
        finally
        {
            try
            {
                lifecycleSmooksResource.getLifecycleResource().remove(smooks) ;
            }
            catch (final LifecycleResourceException lre) {} // ignore
        }
    }

    /**
     * The lifecycle resource factory
     * @author kevin
     */
    private static class SmooksResourceFactory implements LifecycleResourceFactory<IdentityHashMap<Smooks, Smooks>>
    {
        /**
         * Create a resource object which will be associated with the specified lifecycle identity.
         * @param lifecycleIdentity The associated lifecycle identity.
         * @return The lifecycle resource
         * @throws LifecycleResourceException for errors during construction.
         */
        public IdentityHashMap<Smooks, Smooks> createLifecycleResource(final String lifecycleIdentity)
            throws LifecycleResourceException
        {
            return new IdentityHashMap<Smooks, Smooks>() ;
        }

        /**
         * Destroy a resource object which is associated with the specified lifecycle identity.
         * @param resource The lifecycle resource.
         * @param lifecycleIdentity The associated lifecycle identity.
         * @return The lifecycle resource.
         * @throws LifecycleResourceException for errors during destroy.
         */
        public void destroyLifecycleResource(final IdentityHashMap<Smooks, Smooks> resource, final String lifecycleIdentity)
            throws LifecycleResourceException
        {
            if (resource.size() > 0)
            {
                LOGGER.warn("Closing smooks resources for identity " + lifecycleIdentity) ;
                final Iterator<Smooks> smooksIter = resource.keySet().iterator() ;
                while(smooksIter.hasNext())
                {
                    final Smooks smooks = smooksIter.next() ;
                    smooksIter.remove() ;
                    try
                    {
                        smooks.close() ;
                    }
                    catch (final Exception ex)
                    {
                        LOGGER.warn("Unexpected exception closing smooks resource", ex) ;
                    }
                }
            }
        }
    }

    private static class DeploymentApplicationContext extends StandaloneApplicationContext
    {
        private  ClassLoader classLoader ;
        
        public DeploymentApplicationContext(final boolean registerInstalledResources)
        {
            super(registerInstalledResources) ;
        }
        
        @Override
        public void setClassLoader(final ClassLoader classLoader)
        {
            this.classLoader = classLoader ;
        }
        
        @Override
        public ClassLoader getClassLoader()
        {
            final ClassLoader result ;
            if (classLoader != null)
            {
                result = classLoader ;
            }
            else
            {
                final ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader() ;
                if (contextClassLoader != null)
                {
                    result = contextClassLoader ;
                }
                else
                {
                    result = getClass().getClassLoader() ;
                }
            }
            return result ;
        }
    }
}
