/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.util;

import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.IOException;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

/**
 * Util methods for XPath related operations.
 * 
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>				
 * @since 4.2
 *
 */
public class XPathUtil
{
	private static Logger log = Logger.getLogger( XPathUtil.class );
	
	/**
	 * Parses the passed in xml and creates a Dom Document
	 * 
	 * @param xml
	 * @return Document
	 * @throws SAXException
	 * @throws IOException
	 * @throws ParserConfigurationException
	 */
	public static org.w3c.dom.Document getDocument( String xml ) throws SAXException, IOException, ParserConfigurationException
	{
		if (xml == null) 
			throw new IllegalArgumentException( "xml must not be null" );
		
		ByteArrayInputStream inputStream = null;
		try
		{
			DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
			DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
			inputStream = new ByteArrayInputStream( xml.getBytes("UTF-8") );
			return documentBuilder.parse( inputStream );
		} 
		finally
		{
			close( inputStream );
		}
	}
	
	/**
	 * Will compile the passed in XPath expression and evalutate it
	 * against the passed in doument.
	 * 
	 * @param document			Document that the XPath expression should evaluate upon.
	 * @param xPathExpression	XPath expression
	 * @return Element			The Element found
	 * @throws XPathExpressionException
	 */
	public static Element getNodeFromXPathExpression( Document document, String xPathExpression ) throws XPathExpressionException
	{
		if (document == null)
			throw new IllegalArgumentException( "document must not be null" );
		
		if ( xPathExpression == null )
			return document.getDocumentElement();
		
		Node rootNode = (Node) getQNameFromXPathExpression( document, xPathExpression, XPathConstants.NODE );
		
		return (Element) rootNode;
	}
	
	/**
	 * @param xml				XML string that will be the target of the XPath evaluation
	 * @param xPathExpression	XPath expression
	 * @return Element			The Element found
	 * 
	 * @throws XPathExpressionException
	 * @throws SAXException
	 * @throws IOException
	 * @throws ParserConfigurationException
	 */
	public static Element getNodeFromXPathExpression( String xml, String xPathExpression ) throws XPathExpressionException, SAXException, IOException, ParserConfigurationException
	{
		return getNodeFromXPathExpression( getDocument( xml ), xPathExpression );
	}
	
	/**
	 * All public methods should be able to delegate the actual XPath evaluation
	 * to this method.
	 * 
	 * @param document			Document that the XPath expression should evaluate upon.
	 * 							Must not be null. This caller is responsible for checking for null.
	 * @param xPathExpression	XPath expression
	 * 							Must not be null. This caller is responsible for checking for null.
	 * @param type				The XPathConstant type to use. The affects the type returned from the evaluate method.
	 * 
	 * @return Object			of type specified in the passed in <code>type</code> argument
	 * 
	 * @throws XPathExpressionException
	 */
	private static Object getQNameFromXPathExpression( Document document, String xPathExpression, QName type ) throws XPathExpressionException
	{
		XPath xpath = XPathFactory.newInstance().newXPath();
		XPathExpression expression = xpath.compile( xPathExpression );
		
		return expression.evaluate( document.getDocumentElement(), type );
	}

	private static void close (Closeable c )
	{
		try
		{
			if ( c != null )
				c.close();
		} 
		catch (IOException e)
		{
			log.error(  "Error when closing IOStream", e );
		}
	}

}
